<?php

namespace Drupal\nonce_generator\EventSubscriber;

use Drupal\nonce_generator\NonceService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to inject nonces into Content-Security-Policy headers.
 */
class ResponseSubscriber implements EventSubscriberInterface {

  /**
   * The nonce service.
   *
   * @var \Drupal\nonce_generator\NonceService
   */
  protected NonceService $nonceService;

  /**
   * Constructs a new ResponseSubscriber.
   *
   * @param \Drupal\nonce_generator\NonceService $nonce_service
   *   The nonce service.
   */
  public function __construct(NonceService $nonce_service) {
    $this->nonceService = $nonce_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run with a low priority to ensure we run after other modules
    // that might set CSP headers (like SecKit).
    $events[KernelEvents::RESPONSE][] = ['onResponse', -100];
    return $events;
  }

  /**
   * Injects nonce into Content-Security-Policy headers.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    $response = $event->getResponse();
    $nonce = $this->nonceService->getCurrentNonce();

    // Check for both CSP headers.
    $csp_headers = [
      'Content-Security-Policy',
      'Content-Security-Policy-Report-Only',
    ];

    foreach ($csp_headers as $header_name) {
      if ($response->headers->has($header_name)) {
        $csp = $response->headers->get($header_name);
        if (is_string($csp)) {
          $updated_csp = $this->injectNonceIntoScriptSrc($csp, $nonce);
          $response->headers->set($header_name, $updated_csp);
        }
      }
    }

    // If no CSP header exists, create a basic one with our nonce.
    if (!$response->headers->has('Content-Security-Policy') &&
        !$response->headers->has('Content-Security-Policy-Report-Only')) {
      $basic_csp = sprintf("script-src 'self' 'nonce-%s'", $nonce);
      $response->headers->set('Content-Security-Policy', $basic_csp);
    }
  }

  /**
   * Injects nonce into the script-src directive of a CSP header.
   *
   * @param string $csp
   *   The existing CSP header value.
   * @param string $nonce
   *   The nonce to inject.
   *
   * @return string
   *   The updated CSP header value.
   */
  protected function injectNonceIntoScriptSrc(string $csp, string $nonce): string {
    $nonce_directive = sprintf("'nonce-%s'", $nonce);

    // Check if script-src directive exists.
    if (preg_match('/script-src\s+([^;]+)/i', $csp, $matches)) {
      $script_src = trim($matches[1]);

      // Check if nonce is already present to avoid duplicates.
      if (strpos($script_src, $nonce_directive) === FALSE) {
        $updated_script_src = $script_src . ' ' . $nonce_directive;
        $csp = preg_replace(
          '/script-src\s+[^;]+/i',
          "script-src $updated_script_src",
          $csp
        );
      }
    }
    else {
      // No script-src directive exists, add one.
      $script_src_directive = sprintf("script-src 'self' %s", $nonce_directive);
      $csp = trim($csp);
      if (!empty($csp)) {
        $csp .= '; ' . $script_src_directive;
      }
      else {
        $csp = $script_src_directive;
      }
    }

    return $csp;
  }

}
