<?php

namespace Drupal\nonce_generator;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\nonce_generator\Plugin\NonceScriptPluginManager;

/**
 * Lazy builder for nonce script plugins.
 *
 * Provides lazy builder callbacks for rendering nonce script plugins
 * with fresh nonces on each request.
 */
class NonceScriptLazyBuilder implements TrustedCallbackInterface {

  /**
   * The nonce service.
   *
   * @var \Drupal\nonce_generator\NonceService
   */
  protected NonceService $nonceService;

  /**
   * The nonce script plugin manager.
   *
   * @var \Drupal\nonce_generator\Plugin\NonceScriptPluginManager
   */
  protected NonceScriptPluginManager $pluginManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Constructs a NonceScriptLazyBuilder object.
   *
   * @param \Drupal\nonce_generator\NonceService $nonce_service
   *   The nonce service.
   * @param \Drupal\nonce_generator\Plugin\NonceScriptPluginManager $plugin_manager
   *   The nonce script plugin manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(NonceService $nonce_service, NonceScriptPluginManager $plugin_manager, LoggerChannelFactoryInterface $logger_factory) {
    $this->nonceService = $nonce_service;
    $this->pluginManager = $plugin_manager;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['buildScript', 'buildAllScripts'];
  }

  /**
   * Lazy builder callback for rendering a specific nonce script plugin.
   *
   * @param string $plugin_id
   *   The plugin ID to render.
   *
   * @return array
   *   A render array containing the script with nonce.
   */
  public function buildScript(string $plugin_id): array {
    try {
      /** @var \Drupal\nonce_generator\Plugin\NonceScript\NonceScriptPluginInterface $plugin */
      $plugin = $this->pluginManager->createInstance($plugin_id);

      if (!$plugin->isApplicable()) {
        return [];
      }

      $nonce = $this->nonceService->getCurrentNonce();
      $script_content = $plugin->getScript($nonce);

      return [
        '#type' => 'inline_template',
        '#template' => '{{ script|raw }}',
        '#context' => ['script' => $script_content],
        '#cache' => ['max-age' => 0],
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('nonce_generator')->error(
        'Error rendering nonce script plugin @plugin_id: @message',
        [
          '@plugin_id' => $plugin_id,
          '@message' => $e->getMessage(),
        ]
      );
      return [];
    }
  }

  /**
   * Lazy builder callback for rendering all applicable nonce script plugins.
   *
   * @return array
   *   A render array containing all applicable scripts with nonces.
   */
  public function buildAllScripts(): array {
    $build = [];
    $nonce = $this->nonceService->getCurrentNonce();

    foreach ($this->pluginManager->getApplicablePlugins() as $plugin_id => $plugin) {
      try {
        $script_content = $plugin->getScript($nonce);

        $build[$plugin_id] = [
          '#type' => 'inline_template',
          '#template' => '{{ script|raw }}',
          '#context' => ['script' => $script_content],
          '#cache' => ['max-age' => 0],
        ];
      }
      catch (\Exception $e) {
        $this->loggerFactory->get('nonce_generator')->error(
          'Error rendering nonce script plugin @plugin_id: @message',
          [
            '@plugin_id' => $plugin_id,
            '@message' => $e->getMessage(),
          ]
        );
      }
    }

    return $build;
  }

}
