<?php

namespace Drupal\nonce_generator\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\nonce_generator\Plugin\NonceScript\NonceScriptPluginInterface;

/**
 * Plugin manager for nonce script plugins.
 *
 * Manages the discovery and instantiation of nonce script plugins
 * that generate script tags with Content Security Policy nonces.
 */
class NonceScriptPluginManager extends DefaultPluginManager {

  /**
   * Constructs a NonceScriptPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/NonceScript',
      $namespaces,
      $module_handler,
      NonceScriptPluginInterface::class,
      'Drupal\nonce_generator\Annotation\NonceScript'
    );
    $this->alterInfo('nonce_script_info');
    $this->setCacheBackend($cache_backend, 'nonce_script_plugins');
  }

  /**
   * Gets applicable plugin instances, sorted by weight.
   *
   * @return \Drupal\nonce_generator\Plugin\NonceScript\NonceScriptPluginInterface[]
   *   Array of applicable plugin instances, keyed by plugin ID.
   */
  public function getApplicablePlugins(): array {
    $plugins = [];

    foreach ($this->getDefinitions() as $plugin_id => $definition) {
      /** @var \Drupal\nonce_generator\Plugin\NonceScript\NonceScriptPluginInterface $plugin */
      $plugin = $this->createInstance($plugin_id);

      if ($plugin->isApplicable()) {
        $plugins[$plugin_id] = $plugin;
      }
    }

    // Sort by weight.
    uasort($plugins, static function (NonceScriptPluginInterface $a, NonceScriptPluginInterface $b): int {
      return $a->getWeight() <=> $b->getWeight();
    });

    return $plugins;
  }

}
