<?php

namespace Drupal\nonce_generator\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Nonce Generator settings.
 */
class NonceGeneratorConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nonce_generator_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['nonce_generator.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('nonce_generator.settings');

    $form['performance_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Performance Settings'),
      '#description' => $this->t('Configure performance optimizations for the nonce generator.'),
    ];

    $form['performance_settings']['disable_kill_switch'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable cache prevention for better performance'),
      '#description' => $this->t('Allow pages with nonces to be cached for improved performance. Nonces are unique tokens added to Content Security Policy (CSP) headers to allow specific inline scripts to execute safely. When enabled, pages can be cached normally while the same nonce is reused across cached views. This only affects page-level caching - other caching layers like Twig template cache, block cache, and render cache continue to work normally.'),
      '#default_value' => $config->get('disable_kill_switch'),
    ];

    $form['performance_settings']['warning'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--warning']],
      'content' => [
        '#markup' => $this->t('<strong>Security Warning:</strong> Disabling the kill switch allows page caching but nonces will be reused across cached page views. This is suitable for non-critical inline scripts but should be avoided if nonces are used for security-critical functionality that requires unique values per request.'),
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('nonce_generator.settings')
      ->set('disable_kill_switch', $form_state->getValue('disable_kill_switch'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
