<?php

namespace Drupal\nonce_generator;

/**
 * Service for generating and managing CSP nonces per request.
 */
class NonceService {

  /**
   * The current request's nonce.
   *
   * @var string|null
   */
  protected ?string $nonce = NULL;

  /**
   * Gets the current request's nonce, generating it if needed.
   *
   * @return string
   *   The nonce for the current request.
   */
  public function getCurrentNonce(): string {
    if ($this->nonce === NULL) {
      $this->nonce = $this->generateNonce();
    }
    return $this->nonce;
  }

  /**
   * Generates a cryptographically secure nonce.
   *
   * @return string
   *   A base64-encoded nonce suitable for CSP headers.
   *
   * @throws \Exception
   *   When random_bytes() fails to generate random data.
   */
  protected function generateNonce(): string {
    return base64_encode(random_bytes(16));
  }

  /**
   * Resets the nonce for the current request.
   *
   * This method is primarily for testing purposes.
   */
  public function resetNonce(): void {
    $this->nonce = NULL;
  }

}
