<?php

namespace Drupal\nonce_generator\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\nonce_generator\NonceService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to inject nonces into Content-Security-Policy headers.
 */
class ResponseSubscriber implements EventSubscriberInterface {

  /**
   * The nonce service.
   *
   * @var \Drupal\nonce_generator\NonceService
   */
  protected NonceService $nonceService;

  /**
   * The page cache kill switch service.
   *
   * @var \Drupal\Core\PageCache\ResponsePolicy\KillSwitch
   */
  protected KillSwitch $killSwitch;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new ResponseSubscriber.
   *
   * @param \Drupal\nonce_generator\NonceService $nonce_service
   *   The nonce service.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $kill_switch
   *   The page cache kill switch service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(NonceService $nonce_service, KillSwitch $kill_switch, ConfigFactoryInterface $config_factory) {
    $this->nonceService = $nonce_service;
    $this->killSwitch = $kill_switch;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run with a low priority to ensure we run after other modules
    // that might set CSP headers (like SecKit).
    $events[KernelEvents::RESPONSE][] = ['onResponse', -100];
    return $events;
  }

  /**
   * Injects nonce into Content-Security-Policy headers.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    // Conditionally disable page caching based on configuration.
    $config = $this->configFactory->get('nonce_generator.settings');

    if (!$config->get('disable_kill_switch')) {
      $this->killSwitch->trigger();
    }

    $response = $event->getResponse();
    $nonce = $this->nonceService->getCurrentNonce();

    // Check for both CSP headers.
    $csp_headers = [
      'Content-Security-Policy',
      'Content-Security-Policy-Report-Only',
    ];

    foreach ($csp_headers as $header_name) {
      if ($response->headers->has($header_name)) {
        $csp = $response->headers->get($header_name);
        if (is_string($csp)) {
          $updated_csp = $this->injectNonceIntoScriptSrc($csp, $nonce);
          if ($updated_csp) {
            $response->headers->set($header_name, $updated_csp);
          }
        }
      }
    }
  }

  /**
   * Injects nonce into the script-src directive of a CSP header.
   *
   * @param string $csp
   *   The existing CSP header value.
   * @param string $nonce
   *   The nonce to inject.
   *
   * @return string|null
   *   The updated CSP header value, or NULL if no update needed.
   */
  protected function injectNonceIntoScriptSrc(string $csp, string $nonce): ?string {
    $nonce_directive = sprintf("'nonce-%s'", $nonce);

    // Check if script-src directive exists.
    if (preg_match('/script-src\s+([^;]+)/i', $csp, $matches)) {
      $script_src = trim($matches[1]);

      // Skip if unsafe-inline is present.
      if (preg_match("/['\"]unsafe-inline['\"]/i", $script_src)) {
        return NULL;
      }

      // Check if nonce is already present to avoid duplicates.
      if (!str_contains($script_src, $nonce_directive)) {
        $updated_script_src = $script_src . ' ' . $nonce_directive;
        return preg_replace(
          '/script-src\s+[^;]+/i',
          "script-src $updated_script_src",
          $csp
        );
      }
    }
    else {
      // No script-src directive exists, add one.
      $script_src_directive = sprintf("script-src 'self' %s", $nonce_directive);
      $csp = trim($csp);
      if (!empty($csp)) {
        return $csp . '; ' . $script_src_directive;
      }
      else {
        return $script_src_directive;
      }
    }

    return NULL;
  }

}
