<?php

namespace Drupal\nonce_generator\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Nonce Generator settings.
 */
class NonceGeneratorConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nonce_generator_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['nonce_generator.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('nonce_generator.settings');

    $form['performance_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Performance Settings'),
      '#description' => $this->t('Configure performance optimizations for the nonce generator.'),
    ];

    $form['performance_settings']['disable_kill_switch'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow caching with nonces (reuse nonces across requests)'),
      '#description' => $this->t('Enable page caching to improve site performance. Nonces are unique tokens added to Content Security Policy (CSP) headers that allow specific inline scripts to run safely. When this option is enabled, the same nonce will be reused across multiple page views from cache. This only affects full page caching - other caches like Twig templates, blocks, and render cache continue working normally.'),
      '#default_value' => $config->get('disable_kill_switch'),
    ];

    $form['csp_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Security Policy Behavior'),
      '#description' => $this->t('This module only enhances existing CSP headers and will not create new ones. Nonces are automatically skipped when \'unsafe-inline\' is present in the script-src directive, as nonces provide no additional security benefit when unsafe-inline is allowed.'),
    ];

    $form['performance_settings']['warning'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--warning']],
      'content' => [
        '#markup' => $this->t('<strong>Security Note:</strong> When enabled, nonces will be reused across cached page requests instead of being unique per request. Consider whether your security model requires truly unique nonces for every page load before enabling this option.'),
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('nonce_generator.settings')
      ->set('disable_kill_switch', $form_state->getValue('disable_kill_switch'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
