(function notificationMessageDismiss($, Drupal, once) {
  /**
   * Convert document.cookie string to JSON object.
   *
   * @return {Object}
   *   An object containing all cookies as key-value pairs.
   */
  function cookieToJSON() {
    return Object.fromEntries(
      document.cookie
        .split('; ')
        .map((cookie) => cookie.split('='))
        .map(([key, value]) => [key, decodeURIComponent(value)]),
    );
  }

  /**
   * Set a cookie with given name, value and optional days until expiration.
   *
   * @param {string} name
   *   The cookie name.
   * @param {string} value
   *   The cookie value.
   * @param {number} [days]
   *   Optional number of days until cookie expires.
   */
  function setCookie(name, value, days) {
    let expires = '';
    if (days) {
      const date = new Date();
      date.setTime(date.getTime() + days * 24 * 60 * 60 * 1000);
      expires = `; expires=${date.toUTCString()}`;
    }
    document.cookie = `${name}=${encodeURIComponent(value)}${expires}; path=/`;
  }

  /**
   * Read cookie data.
   *
   * @param {string} name
   *   The cookie name.
   *
   * @return {any}
   *   The parsed cookie data or null if not found.
   */
  function readCookieData(name) {
    const cookies = cookieToJSON();
    const cookieData = cookies[name] || null;
    if (typeof cookieData === 'string') {
      try {
        return JSON.parse(cookieData);
      } catch (e) {
        return cookieData;
      }
    }
    return cookieData;
  }

  /**
   * Merge cookie data.
   *
   * @param {string} name
   *   The cookie name.
   * @param {*} data
   *   The cookie data.
   */
  function mergeCookieData(name, data) {
    const cookieData = readCookieData(name) || [];
    const mergedData = [data].concat(cookieData);

    // Remove duplicates and store
    const uniqueData = mergedData.filter(
      function filterUnique(element, index, self) {
        return index === self.indexOf(element);
      },
    );

    setCookie(name, JSON.stringify(uniqueData), 365); // Set cookie to expire in 1 year
  }

  Drupal.behaviors.notificationMessage = {
    attach(context) {
      once('notificationMessage', 'html', context).forEach(
        function processNotifications() {
          const cookieData = readCookieData('notificationMessagesClosed') || [];
          $('.notification-messages .message').each(function processMessage() {
            const messageId = $(this).data('message-id');
            if ($.inArray(messageId, cookieData) === -1) {
              const parentEl = $(this).parent()[0];
              if (parentEl && getComputedStyle(parentEl).display === 'none') {
                $(this).parent().show();
              }
              $(this).show();
            } else {
              $(this).remove();
            }
          });
        },
      );

      $('.notification-messages .message a.message__close', context).click(
        function handleClose(event) {
          event.preventDefault();
          const $parent = $(this).parent();
          const messageId = $parent.data('message-id');
          const notificationCount = $('.notification-messages > div').length;

          const announcement = document.getElementById(
            'notification-message-aria-announcement',
          );
          if (announcement) {
            announcement.textContent = 'Notification message closed';
          }

          if (notificationCount > 1) {
            $parent.remove();
          } else {
            $('.notification-messages').remove();
          }

          if (messageId !== undefined) {
            mergeCookieData('notificationMessagesClosed', messageId);
          }
        },
      );
    },
  };
})(jQuery, Drupal, once);
