<?php

declare(strict_types=1);

namespace Drupal\notification_message\Entity;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityPublishedInterface;

/**
 * Define the notification message interface.
 */
interface NotificationMessageInterface extends ContentEntityInterface, EntityChangedInterface, EntityPublishedInterface {

  /**
   * Default value callback for 'uid' base field definition.
   *
   * @return array
   *   An array of current user default values.
   *
   * @see ::baseFieldDefinitions()
   */
  public static function getCurrentUserId();

  /**
   * Render the notification message.
   *
   * @param string $view_mode
   *   The entity view mode.
   * @param null $langcode
   *   The entity language code.
   *
   * @return array
   *   The render array for the notification message.
   */
  public function view($view_mode = 'full', $langcode = NULL);

  /**
   * Get the notification message author user.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The notification message user entity.
   */
  public function getAuthorUser(): ?EntityInterface;

  /**
   * Get publish end date format.
   *
   * @param string $format
   *   The date format.
   *
   * @return false|string
   *   The publish end date in the specified format.
   */
  public function getPublishEndDateFormat($format);

  /**
   * Get publish start date format.
   *
   * @param string $format
   *   The date format.
   *
   * @return false|string
   *   The publish start date in the specified format.
   */
  public function getPublishStartDateFormat($format);

  /**
   * Get the bundle entity type entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @return \Drupal\notification_message\Entity\NotificationMessageType|null
   *   The entity bundle type entity instance.
   */
  public function getBundleEntityTypeEntity(): ?NotificationMessageType;

  /**
   * Get the notification message configurations.
   *
   * @return array
   *   An array of conditions.
   */
  public function getConditions();

  /**
   * Has notification message conditions.
   *
   * @return bool
   *   Return TRUE if the notification message has conditions; otherwise FALSE.
   */
  public function hasConditions();

  /**
   * Determine if all conditions are required.
   *
   * @return bool
   *   Return TRUE if all conditions are required; otherwise FALSE.
   */
  public function conditionsRequired();

  /**
   * Evaluate notification message conditions.
   *
   * @param array $contexts
   *   An array of contexts to pass to condition plugins.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *
   * @return bool
   *   Return TRUE if the condition evaluated; otherwise FALSE.
   */
  public function evaluateConditions(array $contexts = []);

}
