# Notification Server

A Drupal module that provides real-time notification capabilities by integrating with an external notification server through HTTP and WebSocket protocols.

!!! important "Developer Module"
    This is a **developer module** that provides APIs and services for integrating real-time notifications into your Drupal application. It will not do anything on its own and requires custom code or other modules to utilize its functionality.

## Overview

The Notification Server module enables real-time notifications in Drupal applications by connecting to an external notification server. This module serves as a client interface to the [Notification Server](https://github.com/d34dman/notification-server) project and provides:

!!! note "Self-Hosted Alternative"
    The Notification Server is a **self-hosted solution** that provides an alternative to SaaS-based real-time messaging services like Pusher, Firebase, or Ably. By running your own notification server, you maintain full control over your data, avoid vendor lock-in, and can customize the solution to meet your specific needs.

1. **HTTP API Client** - For publishing notifications and managing channels
2. **WebSocket Integration** - For real-time bidirectional communication with browser clients
3. **Channel Management** - For creating channels and managing access control
4. **Client Management** - For handling WebSocket client connections

## Use Cases

This module is particularly useful for:

- Real-time updates in web applications
- Push notifications to specific channels
- Live data streaming
- Chat applications
- Real-time dashboards
- Content moderation workflows
- User activity notifications

## Quick Start

### Basic Usage

```php
<?php
// Get the notification client service
$notification_client = \Drupal::service('notification_server.client');

// Send a simple notification
$notification_client->publishNotification('channel_name', 'Hello, World!');

// Send a structured notification
$notification_client->publishNotification('user_notifications', [
  'type' => 'info',
  'message' => 'New content available',
  'data' => [
    'contentId' => 123,
    'title' => 'Updated Article',
  ],
]);
```

### WebSocket Integration

1. **Generate a client ID** (server-side):
   ```php
   <?php
   $client_id = \Drupal::service('notification_server.client')->generateClientId();
   ```

2. **Connect to WebSocket server** (client-side):
   ```javascript
   const websocketUrl = drupalSettings.notificationServer.websocketUrl;
   const socket = new WebSocket(`${websocketUrl}?clientId=${clientId}`);
   ```

3. **Subscribe to a channel**:
   ```javascript
   socket.send(JSON.stringify({
     type: 'subscribe',
     clientId: clientId,
     channel: 'channel_name'
   }));
   ```

4. **Listen for messages**:
   ```javascript
   socket.onmessage = function(event) {
     const data = JSON.parse(event.data);
     if (data.type === 'notification') {
       console.log('Received:', data.data);
     }
   };
   ```

## Getting Started

1. Review the [Requirements](Requirements.md) to ensure your system is compatible
2. Check [Dependencies](Dependencies.md) to set up the external notification server
3. Follow the installation instructions in [Quick Reference](QuickReference.md)
4. Explore the [Features](Features.md) to understand what's available
5. Use the [API Reference](APIReference.md) for detailed method documentation

## Demo Module

The optional `notification_server_demo` module provides demo interfaces for testing:

- **HTTP Demo**: `/admin/config/system/notification-server/http-demo`
- **WebSocket Demo**: `/admin/config/system/notification-server/websocket-demo`

See [Notification Server Demo](modules/notification_server_demo.md) for more details.

## Support

- **Issue Queue**: [Project Issues](https://www.drupal.org/project/issues/notification_server) (when available on Drupal.org)
- **Notification Server Issues**: [GitHub Issues](https://github.com/d34dman/notification-server/issues) for server-related problems

## Related Projects

- **[Notification Server](https://github.com/d34dman/notification-server)** - The external server this module connects to
- **[DDEV Notification Server Addon](https://addons.ddev.com/addons/d34dman/ddev-notification-server)** - Easy DDEV integration
