/**
 * @file
 * JavaScript behaviors for WebSocket notification demo.
 */

(function ($, Drupal, once) {
  'use strict';

  const MAX_LOG_ENTRIES = 100;

  /**
   * WebSocket demo behavior.
   */
  Drupal.behaviors.notificationWebSocketDemo = {
    websocket: null,
    clientId: null,
    subscribedChannels: new Set(),

    attach: function (context, settings) {
      once('notification-websocket-demo', '.websocket-demo', context).forEach(() => {
        this.initializeInterface();
      });
    },

    /**
     * Initialize the user interface.
     */
    initializeInterface: function () {
      const container = document.querySelector('.websocket-demo');

      // Add client ID form
      const clientSection = document.getElementById('client-management');
      clientSection.innerHTML = `
        <div class="client-form">
          <h3 class="client-form__title">${Drupal.t('Client Management')}</h3>
          <div class="client-form__field">
            <input type="text" class="client-form__input" id="client-id-input" placeholder="${Drupal.t('Enter client ID')}" />
            <button type="button" id="connect-btn" class="client-form__button client-form__button--primary">${Drupal.t('Connect')}</button>
            <button type="button" id="disconnect-btn" class="client-form__button client-form__button--secondary" disabled>${Drupal.t('Disconnect')}</button>
          </div>
        </div>
      `;

      // Add subscription form
      const subscriptionSection = document.getElementById('subscription-management');
      subscriptionSection.innerHTML = `
        <div class="subscription-form">
          <h3 class="subscription-form__title">${Drupal.t('Channel Subscription')}</h3>
          <div class="subscription-form__field">
            <input type="text" class="subscription-form__input" id="channel-input" placeholder="${Drupal.t('Enter channel name')}" disabled />
            <button type="button" id="subscribe-btn" class="subscription-form__button subscription-form__button--primary" disabled>${Drupal.t('Subscribe')}</button>
            <button type="button" id="unsubscribe-btn" class="subscription-form__button subscription-form__button--secondary" disabled>${Drupal.t('Unsubscribe')}</button>
          </div>
        </div>
      `;

      // Add notification info message
      const notificationSection = document.getElementById('notification-sending');
      notificationSection.innerHTML = `
        <div class="info-panel">
          <h3 class="info-panel__title">${Drupal.t('Sending Notifications')}</h3>
          <p class="info-panel__text">${Drupal.t('Notifications can only be set through the server-side API. This is by design.\
            Websockets are capable of bi-directional communication. However, only\
            subscribing and un-subscribing actions can be initiated from client side. Every other write operation\
            has to be done via Drupal.\
            Use <b>HTTP Demo</b> to send notification. The notification can be send from a different user or from an entirely different browser.')}</p>
          <p class="info-panel__text">${Drupal.t('<b>Note:</b> To programmatically send notification, use <code>notification_server.client</code> service to publish messages to channels.')}</p>
          <pre class="info-panel__code"><code class="info-panel__code-text">$notification_client = \\Drupal::service('notification_server.client');
$notification_client->publish('channel_name', 'Your message here');</code></pre>
        </div>
      `;

      // Initialize subscribed channels list
      const channelsSection = document.getElementById('subscribed-channels');
      channelsSection.innerHTML = `
        <div class="channel-list">
          <h3 class="channel-list__title">${Drupal.t('Subscribed Channels')}</h3>
          <ul class="channel-list__items" id="channels-list"></ul>
        </div>
      `;

      // Initialize notification log
      const logSection = document.getElementById('notification-log');
      logSection.innerHTML = `
        <div class="notification-log">
          <h3 class="notification-log__title">${Drupal.t('Notification Log')}</h3>
          <div class="notification-log__messages"></div>
        </div>
      `;

      // Add event listeners
      document.getElementById('connect-btn').addEventListener('click', () => this.handleConnect());
      document.getElementById('disconnect-btn').addEventListener('click', () => this.handleDisconnect());
      document.getElementById('subscribe-btn').addEventListener('click', () => this.handleSubscribe());
      document.getElementById('unsubscribe-btn').addEventListener('click', () => this.handleUnsubscribe());

      // Initialize connection status
      this.updateConnectionStatus('disconnected', Drupal.t('Not connected'));
    },

    /**
     * Initialize WebSocket connection.
     *
     * @param {string} websocketUrl
     *   The WebSocket server URL.
     */
    initializeWebSocket: function (websocketUrl) {
      if (!websocketUrl) {
        this.updateConnectionStatus('error', Drupal.t('WebSocket URL not configured'));
        return;
      }

      if (!this.clientId) {
        return;
      }

      try {
        // Append client ID as URL parameter
        const wsUrl = `${websocketUrl}?clientId=${encodeURIComponent(this.clientId)}`;
        this.websocket = new WebSocket(wsUrl);
        this.setupWebSocketHandlers();
      }
      catch (error) {
        this.updateConnectionStatus('error', Drupal.t('Failed to initialize WebSocket connection'));
        console.error('WebSocket initialization error:', error);
      }
    },

    /**
     * Set up WebSocket event handlers.
     */
    setupWebSocketHandlers: function () {
      this.websocket.onopen = () => {
        this.updateConnectionStatus('connected', Drupal.t('Connected to WebSocket server'));
      };

      this.websocket.onclose = () => {
        this.updateConnectionStatus('disconnected', Drupal.t('Disconnected from WebSocket server'));
      };

      this.websocket.onerror = (error) => {
        this.updateConnectionStatus('error', Drupal.t('WebSocket error occurred'));
        console.error('WebSocket error:', error);
      };

      this.websocket.onmessage = (event) => {
        try {
          const data = JSON.parse(event.data);
          this.handleIncomingMessage(data);
        }
        catch (error) {
          console.error('Error parsing message:', error);
          this.showMessage(Drupal.t('Error processing received message'), 'error');
        }
      };
    },

    /**
     * Handle incoming WebSocket messages.
     *
     * @param {Object} data
     *   The parsed message data.
     */
    handleIncomingMessage: function (data) {
      if (data.type === 'notification') {
        this.showMessage(`${Drupal.t('Received message on channel "@channel": @message', {
          '@channel': data.channel,
          '@message': data.data.message
        })}`, 'success');
      }
    },

    /**
     * Handle client connection.
     */
    handleConnect: function () {
      const clientInput = document.querySelector('.client-form__input');
      const clientId = clientInput.value.trim();

      if (!clientId) {
        this.showMessage(Drupal.t('Please enter a client ID'), 'error');
        return;
      }

      this.clientId = clientId;
      this.enableSubscriptionControls();
      document.getElementById('connect-btn').disabled = true;
      document.getElementById('disconnect-btn').disabled = false;
      clientInput.disabled = true;

      // Initialize WebSocket with the client ID
      this.initializeWebSocket(drupalSettings.notificationServer.websocketUrl);

      this.showMessage(Drupal.t('Connected with client ID: @clientId', {
        '@clientId': clientId
      }), 'success');
    },

    /**
     * Handle client disconnection.
     */
    handleDisconnect: function () {
      // Close existing WebSocket connection if it exists
      if (this.websocket) {
        this.websocket.close();
        this.websocket = null;
      }

      // Unsubscribe from all channels first
      this.subscribedChannels.forEach(channel => {
        this.sendUnsubscription(channel);
      });

      this.clientId = null;
      this.subscribedChannels.clear();
      this.updateChannelsList();
      this.disableSubscriptionControls();

      const clientInput = document.querySelector('.client-form__input');
      document.getElementById('connect-btn').disabled = false;
      document.getElementById('disconnect-btn').disabled = true;
      clientInput.disabled = false;
      clientInput.value = '';

      this.showMessage(Drupal.t('Disconnected client'), 'info');
    },

    /**
     * Enable subscription controls.
     */
    enableSubscriptionControls: function () {
      document.querySelector('.subscription-form__input').disabled = false;
      document.getElementById('subscribe-btn').disabled = false;
      document.getElementById('unsubscribe-btn').disabled = false;
    },

    /**
     * Disable subscription controls.
     */
    disableSubscriptionControls: function () {
      const channelInput = document.querySelector('.subscription-form__input');
      channelInput.disabled = true;
      channelInput.value = '';
      document.getElementById('subscribe-btn').disabled = true;
      document.getElementById('unsubscribe-btn').disabled = true;
    },

    /**
     * Handle channel subscription.
     */
    handleSubscribe: function () {
      if (!this.clientId) {
        this.showMessage(Drupal.t('Please connect with a client ID first'), 'error');
        return;
      }

      const channelInput = document.querySelector('.subscription-form__input');
      const channel = channelInput.value.trim();

      if (!channel) {
        this.showMessage(Drupal.t('Please enter a channel name'), 'error');
        return;
      }

      if (this.subscribedChannels.has(channel)) {
        this.showMessage(Drupal.t('Already subscribed to channel "@channel"', {
          '@channel': channel
        }), 'error');
        return;
      }

      this.sendSubscription(channel);
      this.subscribedChannels.add(channel);
      this.updateChannelsList();
      channelInput.value = '';

      this.showMessage(Drupal.t('Subscribed to channel "@channel"', {
        '@channel': channel
      }), 'success');
    },

    /**
     * Handle channel unsubscription.
     */
    handleUnsubscribe: function () {
      if (!this.clientId) {
        this.showMessage(Drupal.t('Please connect with a client ID first'), 'error');
        return;
      }

      const channelInput = document.querySelector('.subscription-form__input');
      const channel = channelInput.value.trim();

      if (!channel) {
        this.showMessage(Drupal.t('Please enter a channel name'), 'error');
        return;
      }

      if (!this.subscribedChannels.has(channel)) {
        this.showMessage(Drupal.t('Not subscribed to channel "@channel"', {
          '@channel': channel
        }), 'error');
        return;
      }

      this.sendUnsubscription(channel);
      this.subscribedChannels.delete(channel);
      this.updateChannelsList();
      channelInput.value = '';

      this.showMessage(Drupal.t('Unsubscribed from channel "@channel"', {
        '@channel': channel
      }), 'success');
    },

    /**
     * Send a subscription message to the WebSocket server.
     *
     * @param {string} channel
     *   The channel to subscribe to.
     */
    sendSubscription: function (channel) {
      if (this.websocket && this.websocket.readyState === WebSocket.OPEN) {
        this.websocket.send(JSON.stringify({
          type: 'subscribe',
          clientId: this.clientId,
          channel: channel
        }));
      }
    },

    /**
     * Send an unsubscription message to the WebSocket server.
     *
     * @param {string} channel
     *   The channel to unsubscribe from.
     */
    sendUnsubscription: function (channel) {
      if (this.websocket && this.websocket.readyState === WebSocket.OPEN) {
        this.websocket.send(JSON.stringify({
          type: 'unsubscribe',
          clientId: this.clientId,
          channel: channel
        }));
      }
    },

    /**
     * Update the connection status display.
     *
     * @param {string} status
     *   The connection status.
     * @param {string} message
     *   The status message to display.
     */
    updateConnectionStatus: function (status, message) {
      const statusContainer = document.getElementById('websocket-connection-status');
      statusContainer.className = `connection-status connection-status--${status}`;
      statusContainer.innerHTML = `
        <div class="connection-status__indicator"></div>
        <span class="connection-status__message">${message}</span>
      `;
    },

    /**
     * Update the subscribed channels list display.
     */
    updateChannelsList: function () {
      const channelsList = document.querySelector('.channel-list__items');
      channelsList.innerHTML = '';

      this.subscribedChannels.forEach(channel => {
        const li = document.createElement('li');
        li.className = 'channel-list__item';
        li.textContent = channel;
        channelsList.appendChild(li);
      });
    },

    /**
     * Show a message in the notification log.
     *
     * @param {string} message
     *   The message to show.
     * @param {string} type
     *   The message type (success or error).
     */
    showMessage: function (message, type) {
      const logContainer = document.querySelector('.notification-log__messages');
      const timestamp = new Date().toLocaleTimeString();

      const messageElement = document.createElement('div');
      messageElement.className = `notification-log__item notification-log__item--${type}`;
      messageElement.innerHTML = `
        <span class="notification-log__timestamp">[${timestamp}]</span>
        <span class="notification-log__message">${message}</span>
      `;

      logContainer.insertBefore(messageElement, logContainer.firstChild);

      // Limit the number of messages
      while (logContainer.children.length > MAX_LOG_ENTRIES) {
        logContainer.removeChild(logContainer.lastChild);
      }
    }
  };

})(jQuery, Drupal, once);
