# Notification Server Client Service

The `notification_server.client` service provides a robust interface for sending real-time notifications through both HTTP and WebSocket protocols in Drupal applications.

## Service Overview

The NotificationServerClient service is responsible for:
- Publishing messages to specific channels
- Managing channel subscriptions
- Handling client connections
- Error handling and logging

## Basic Usage

### Getting the Service

```php
<?php
// Using dependency injection (recommended)
public function __construct(
  NotificationServerClientInterface $notification_client
) {
  $this->notificationClient = $notification_client;
}

// Or using the service container
$notification_client = \Drupal::service('notification_server.client');
```

### Publishing Messages

```php
<?php
// Simple message publishing
$notification_client->publish('channel_name', 'Hello, World!');

// Publishing with message type
$notification_client->publish('user_notifications', [
  'type' => 'info',
  'message' => 'New content available',
  'data' => [
    'contentId' => 123,
    'title' => 'Updated Article',
  ],
]);

// Broadcasting to multiple channels
$channels = ['general', 'announcements'];
foreach ($channels as $channel) {
  $notification_client->publish($channel, 'Broadcast message');
}
```

### Error Handling

```php
<?php
try {
  $notification_client->publish('channel_name', 'Test message');
}
catch (NotificationException $e) {
  \Drupal::logger('my_module')->error('Failed to send notification: @message', [
    '@message' => $e->getMessage(),
  ]);
}
```

## Advanced Usage

### Custom Message Formatting

```php
<?php
// Structured notification for user updates
$notification_client->publish('user_updates', [
  'type' => 'user_status',
  'message' => t('User @name has updated their profile', [
    '@name' => $user->getDisplayName(),
  ]),
  'data' => [
    'uid' => $user->id(),
    'timestamp' => time(),
    'changes' => [
      'field_name' => 'new_value',
    ],
  ],
]);

// Content moderation notifications
$notification_client->publish('content_moderation', [
  'type' => 'workflow_transition',
  'message' => t('Content "@title" has been @action', [
    '@title' => $node->getTitle(),
    '@action' => $transition->getLabel(),
  ]),
  'data' => [
    'nid' => $node->id(),
    'workflow' => $workflow->id(),
    'from_state' => $from_state->id(),
    'to_state' => $to_state->id(),
    'timestamp' => \Drupal::time()->getCurrentTime(),
  ],
]);
```

### Integration Examples

#### Form Submit Handler

```php
<?php
/**
 * Implements hook_form_submit().
 */
public function submitForm(array &$form, FormStateInterface $form_state) {
  // Process form submission
  $result = $this->processFormData($form_state->getValues());
  
  // Send notification
  try {
    $this->notificationClient->publish('form_submissions', [
      'type' => 'form_submitted',
      'message' => t('New form submission received'),
      'data' => [
        'form_id' => $form['#id'],
        'submission_id' => $result->id(),
        'timestamp' => time(),
      ],
    ]);
  }
  catch (NotificationException $e) {
    $this->logger('my_module')->error('Notification failed: @message', [
      '@message' => $e->getMessage(),
    ]);
  }
}
```

#### Entity Update Hook

```php
<?php
/**
 * Implements hook_entity_update().
 */
function my_module_entity_update(EntityInterface $entity) {
  if ($entity instanceof NodeInterface) {
    /** @var \Drupal\notification_server\Service\NotificationServerClientInterface $notification_client */
    $notification_client = \Drupal::service('notification_server.client');
    
    try {
      $notification_client->publish("node_updates_{$entity->bundle()}", [
        'type' => 'node_updated',
        'message' => t('Content "@title" has been updated', [
          '@title' => $entity->getTitle(),
        ]),
        'data' => [
          'nid' => $entity->id(),
          'type' => $entity->bundle(),
          'changed' => $entity->getChangedTime(),
          'author' => [
            'uid' => $entity->getOwner()->id(),
            'name' => $entity->getOwner()->getDisplayName(),
          ],
        ],
      ]);
    }
    catch (NotificationException $e) {
      \Drupal::logger('my_module')->error('Failed to send node update notification: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }
}
```

#### Queue Worker Integration

```php
<?php
/**
 * A queue worker that sends notifications.
 */
class NotificationQueueWorker extends QueueWorkerBase {

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    /** @var \Drupal\notification_server\Service\NotificationServerClientInterface $notification_client */
    $notification_client = \Drupal::service('notification_server.client');
    
    try {
      $notification_client->publish($data['channel'], [
        'type' => $data['type'],
        'message' => $data['message'],
        'data' => $data['payload'] ?? [],
      ]);
    }
    catch (NotificationException $e) {
      // Log error and throw new exception to retry the item
      \Drupal::logger('my_module')->error('Queue notification failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw new SuspendQueueException($e->getMessage());
    }
  }
}
```

## Best Practices

1. **Error Handling**
   - Always wrap notification calls in try-catch blocks
   - Log errors appropriately
   - Consider fallback mechanisms for critical notifications

2. **Channel Naming**
   - Use consistent naming conventions
   - Consider prefixing channels with module name
   - Use descriptive names that indicate the purpose

3. **Message Structure**
   - Keep messages concise and relevant
   - Include necessary context in data payload
   - Use proper translation for user-facing messages

4. **Performance**
   - Consider batching notifications when possible
   - Use queue workers for non-immediate notifications
   - Monitor notification frequency and volume

5. **Security**
   - Validate input data before sending
   - Use appropriate access checks
   - Don't include sensitive information in notifications

## Troubleshooting

### Common Issues

1. **Connection Failures**
   - Check WebSocket server status
   - Verify server URL configuration
   - Check network connectivity

2. **Message Delivery**
   - Verify channel names
   - Check client subscriptions
   - Monitor server logs

3. **Performance Issues**
   - Review notification frequency
   - Check message payload size
   - Monitor server resources

## API Reference

### Methods

#### `publish()`

```php
<?php
/**
 * Publishes a message to a channel.
 *
 * @param string $channel
 *   The channel name.
 * @param mixed $message
 *   The message to publish (string or array).
 *
 * @throws \Drupal\notification_server\Exception\NotificationException
 *   When the message cannot be published.
 */
public function publish($channel, $message);
```

#### `subscribe()`

```php
<?php
/**
 * Subscribes to a channel.
 *
 * @param string $channel
 *   The channel name.
 *
 * @throws \Drupal\notification_server\Exception\NotificationException
 *   When the subscription fails.
 */
public function subscribe($channel);
```

#### `unsubscribe()`

```php
<?php
/**
 * Unsubscribes from a channel.
 *
 * @param string $channel
 *   The channel name.
 *
 * @throws \Drupal\notification_server\Exception\NotificationException
 *   When the unsubscription fails.
 */
public function unsubscribe($channel);
``` 