# Troubleshooting

This guide helps you diagnose and resolve common issues with the Notification Server module.

## Quick Diagnosis Checklist

!!! tip "Start Here for Any Issue"

    Before diving into specific issues, run through this quick checklist:

    - [ ] Is Redis running and accessible?
    - [ ] Is the notification server running and accessible?
    - [ ] Are the correct URLs configured in Drupal?
    - [ ] Do the required ports have network connectivity?
    - [ ] Are the required permissions granted?

## Connection Issues

### Redis Connection Problems

**Symptoms:**
- Notification server fails to start
- Error messages about Redis connection
- Data not persisting between restarts

**Diagnosis:**

!!! example "Testing Redis Connectivity"

    ```bash
    # Test Redis connectivity
    redis-cli ping
    # Expected output: PONG

    # Check if Redis is running
    docker ps | grep redis
    # or
    sudo systemctl status redis
    ```

**Solutions:**

!!! tip "Redis Troubleshooting Steps"

    1. **Start Redis if not running:**
       ```bash
       # Docker
       docker start redis
       
       # System service
       sudo systemctl start redis
       
       # DDEV
       ddev start
       ```

    2. **Check Redis configuration:**
       ```bash
       # Verify Redis is listening on correct port
       netstat -tlnp | grep 6379
       ```

    3. **Fix connection URL:**
       - Check `REDIS_URL` environment variable
       - Ensure hostname/IP is correct
       - Verify port number (default: 6379)

### Notification Server Connection Issues

**Symptoms:**
- HTTP requests to notification server fail
- WebSocket connections cannot be established
- Drupal module reports connection errors

**Diagnosis:**

!!! example "Testing Notification Server Connectivity"

    ```bash
    # Test HTTP API connectivity
    curl http://localhost:3000/health
    # Expected: {"status":"ok","timestamp":"..."}

    # Test port accessibility
    telnet localhost 3000
    telnet localhost 8080

    # Check if notification server is running
    docker ps | grep notification-server
    ```

**Solutions:**

1. **Start notification server:**
   ```bash
   # Docker Compose
   docker-compose up -d notification-server
   
   # Docker
   docker start notification-server
   
   # Manual
   cd notification-server && npm start
   ```

2. **Check server logs:**
   ```bash
   # Docker
   docker logs notification-server
   
   # Manual installation
   tail -f notification-server/logs/server.log
   ```

3. **Verify environment variables:**
   ```bash
   # Check Docker environment
   docker exec notification-server env | grep -E "(PORT|REDIS_URL)"
   ```

### Network Connectivity Issues

**Symptoms:**
- Drupal can't reach notification server
- WebSocket connections time out
- Intermittent connectivity problems

**Diagnosis:**
```bash
# Test network connectivity from Drupal server
curl -v http://notification-server-host:3000/health

# Check firewall rules
sudo ufw status
sudo iptables -L

# Test WebSocket connectivity
wscat -c ws://notification-server-host:8080
```

**Solutions:**

1. **Configure firewall:**
   ```bash
   # Allow required ports
   sudo ufw allow 3000
   sudo ufw allow 8080
   sudo ufw allow 6379
   ```

2. **Check Docker networking:**
   ```bash
   # Verify Docker networks
   docker network ls
   docker network inspect bridge
   ```

3. **Verify DNS resolution:**
   ```bash
   # Test hostname resolution
   nslookup notification-server-host
   ping notification-server-host
   ```

## Configuration Problems

### Module Not Configured

**Symptoms:**
- Error: "Notification server not configured"
- No response from notification server
- Configuration form shows empty values

**Solutions:**

!!! info "Configuration Steps"

    1. **Configure server URLs:**
       - Visit `/admin/config/system/notification-server`
       - Set **Server URL**: `http://localhost:3000` (or your server URL)
       - Set **WebSocket URL**: `ws://localhost:8080` (or your WebSocket URL)
       - Save configuration

    2. **Verify URL format:**
       - HTTP URL should include protocol: `http://` or `https://`
       - WebSocket URL should use `ws://` or `wss://`
       - Include port numbers if not standard (80/443)

    3. **DDEV specific configuration:**
       - Use `notification-server` as hostname
       - **Server URL**: `http://notification-server:3000`
       - **WebSocket URL**: `ws://notification-server:8080`

### Invalid URLs

**Symptoms:**
- WebSocket URL validation errors
- "Invalid websocket URL" error message
- Configuration form rejects URLs

**Solutions:**

!!! tip "URL Format Reference"

    1. **Check URL format:**
       ```
       Valid WebSocket URLs:
       ✓ ws://localhost:8080
       ✓ wss://secure-server.com:8080
       ✓ ws://192.168.1.100:8080
       
       Invalid WebSocket URLs:
       ✗ http://localhost:8080 (wrong protocol)
       ✗ ws://localhost (missing port)
       ✗ localhost:8080 (missing protocol)
       ```

    2. **Common URL patterns:**
       ```
       Local development: ws://localhost:8080
       DDEV: ws://notification-server:8080
       Production: wss://your-domain.com:8080
       ```

### Environment Variable Issues

**Symptoms:**
- Notification server uses wrong ports
- Redis connection fails despite Redis running
- CORS errors in browser

**Diagnosis:**
```bash
# Check notification server environment
docker exec notification-server env

# Verify expected variables
docker exec notification-server printenv | grep -E "(PORT|REDIS_URL|CORS_ORIGIN)"
```

**Solutions:**

1. **Set required environment variables:**
   ```yaml
   # docker-compose.yml
   environment:
     - PORT=3000
     - WS_PORT=8080
     - REDIS_URL=redis://redis:6379
     - CORS_ORIGIN=*
   ```

2. **Restart after environment changes:**
   ```bash
   docker-compose down
   docker-compose up -d
   ```

## Permission Problems

### Admin Access Issues

**Symptoms:**
- Cannot access configuration page
- "Access denied" when trying to configure module
- Configuration menu items not visible

**Solutions:**

1. **Grant required permissions:**
   - Go to `/admin/people/permissions`
   - Find "Administer notification_server configuration"
   - Grant to appropriate roles

2. **Check user roles:**
   - Ensure user has administrative role
   - Verify role has the required permissions

### Demo Access Issues

**Symptoms:**
- Cannot access demo pages
- Demo menu items not visible
- "Access denied" on demo URLs

**Solutions:**

1. **Grant demo permissions:**
   - `access notification_server http demo`
   - `access notification_server websocket demo`

2. **Enable demo module:**
   ```bash
   drush en notification_server_demo
   ```

## Message Delivery Issues

### Messages Not Publishing

**Symptoms:**
- `publishNotification()` returns null
- No error messages but messages don't appear
- API calls succeed but clients don't receive messages

**Diagnosis:**
```php
<?php
// Enable detailed logging
$logger = \Drupal::logger('notification_server');
$logger->info('Testing notification publish');

$result = $notification_client->publishNotification('test', 'Hello');
if ($result === null) {
  $logger->error('Publish failed');
} else {
  $logger->info('Publish succeeded: @result', ['@result' => json_encode($result)]);
}
```

**Solutions:**

1. **Check channel existence:**
   ```php
   <?php
   // Create channel first
   use Drupal\notification_server\DTO\ChannelDTO;
   $channel = new ChannelDTO('test');
   $notification_client->createChannel($channel);
   ```

2. **Verify server logs:**
   ```bash
   docker logs -f notification-server
   ```

3. **Test with simple message:**
   ```php
   <?php
   $result = $notification_client->publishNotification('test', 'simple string');
   ```

### WebSocket Messages Not Received

!!! warning "Common WebSocket Issue"

    This is often caused by channel access restrictions or invalid client IDs. Check access permissions first.

**Symptoms:**
- WebSocket connection established
- Subscription appears successful
- Published messages not received by clients

**Diagnosis:**

1. **Check client ID validity:**
   ```php
   <?php
   $is_valid = $notification_client->validateClientId($client_id);
   if (!$is_valid) {
     // Generate new client ID
     $client_id = $notification_client->generateClientId();
   }
   ```

2. **Verify subscription:**
   ```javascript
   socket.onmessage = function(event) {
     console.log('Received message:', event.data);
     const data = JSON.parse(event.data);
     console.log('Message type:', data.type);
   };
   ```

**Solutions:**

1. **Grant channel access:**
   ```php
   <?php
   $notification_client->grantChannelAccess('channel_name', $client_id);
   ```

2. **Check subscription format:**
   ```javascript
   // Correct subscription format
   socket.send(JSON.stringify({
     type: 'subscription',
     action: 'subscribe',
     channel: 'channel_name'
   }));
   ```

3. **Monitor WebSocket connection:**
   ```javascript
   socket.onopen = () => console.log('WebSocket connected');
   socket.onclose = () => console.log('WebSocket disconnected');
   socket.onerror = (error) => console.error('WebSocket error:', error);
   ```

### Channel Access Issues

**Symptoms:**
- Clients cannot subscribe to channels
- "Access denied" errors in WebSocket
- Subscription attempts fail silently

**Solutions:**

1. **Create channel with public access:**
   ```php
   <?php
   use Drupal\notification_server\DTO\ChannelDTO;
   use Drupal\notification_server\DTO\ChannelRulesDTO;
   
   $rules = new ChannelRulesDTO(isPublic: true);
   $channel = new ChannelDTO('public_channel', $rules);
   $notification_client->createChannel($channel);
   ```

2. **Grant specific client access:**
   ```php
   <?php
   $notification_client->grantChannelAccess('channel_name', $client_id);
   ```

3. **Use pattern-based access:**
   ```php
   <?php
   $rules = new ChannelRulesDTO(
     allowedPatterns: ['user_.*', 'admin_.*']
   );
   $channel = new ChannelDTO('pattern_channel', $rules);
   $notification_client->createChannel($channel);
   ```

## Performance Issues

### Slow Message Delivery

**Symptoms:**
- Messages take long time to deliver
- High latency in WebSocket communication
- Timeouts during API calls

**Diagnosis:**
```bash
# Check Redis performance
redis-cli --latency-history

# Monitor notification server resources
docker stats notification-server

# Check network latency
ping notification-server-host
```

**Solutions:**

1. **Optimize Redis:**
   ```redis
   # Increase memory
   maxmemory 512mb
   
   # Use faster persistence
   appendfsync no
   ```

2. **Scale notification server:**
   ```yaml
   # Add resource limits
   notification-server:
     deploy:
       resources:
         limits:
           cpus: '2'
           memory: 1G
   ```

3. **Monitor connection limits:**
   ```env
   MAX_CONNECTIONS=2000
   ```

### High Memory Usage

**Symptoms:**
- Redis memory usage grows continuously
- Out of memory errors
- Performance degradation over time

**Solutions:**

1. **Configure Redis memory management:**
   ```redis
   maxmemory 256mb
   maxmemory-policy allkeys-lru
   ```

2. **Set message TTL:**
   ```env
   MESSAGE_TTL=3600  # 1 hour
   ```

3. **Monitor and cleanup:**
   ```bash
   # Check Redis memory usage
   redis-cli info memory
   
   # Cleanup expired keys
   redis-cli --scan --pattern "*" | xargs redis-cli del
   ```

## Browser-Specific Issues

### WebSocket Connection Failures

**Symptoms:**
- WebSocket connections fail in specific browsers
- CORS errors in browser console
- Mixed content warnings

**Solutions:**

1. **Fix CORS configuration:**
   ```env
   CORS_ORIGIN=https://yourdomain.com,http://localhost
   ```

2. **Use secure WebSocket for HTTPS sites:**
   ```javascript
   // Use wss:// for HTTPS sites
   const websocketUrl = location.protocol === 'https:' 
     ? 'wss://your-server.com:8080'
     : 'ws://your-server.com:8080';
   ```

3. **Handle mixed content:**
   - Use HTTPS for all connections in production
   - Configure proper SSL certificates

## Logging and Debugging

### Enable Debug Logging

**Drupal:**
```php
<?php
// In settings.php
$config['system.logging']['error_level'] = 'verbose';
```

**Notification Server:**
```env
LOG_LEVEL=debug
```

**Redis:**
```redis
loglevel verbose
logfile /var/log/redis/redis-server.log
```

### Monitor Logs

```bash
# Watch Drupal logs
tail -f /var/log/drupal/drupal.log

# Watch notification server logs
docker logs -f notification-server

# Watch Redis logs
docker logs -f redis

# Watch all logs together
docker-compose logs -f
```

### Debug Tools

**WebSocket Testing:**
```bash
# Install wscat
npm install -g wscat

# Test WebSocket connection
wscat -c ws://localhost:8080?clientId=test123
```

**HTTP API Testing:**
```bash
# Test notification publishing
curl -X POST http://localhost:3000/api/notifications \
  -H "Content-Type: application/json" \
  -d '{"channel":"test","message":"Hello World"}'
```

## Getting Help

### Collect Debug Information

!!! info "Information to Provide When Seeking Help"

    When seeking help, provide:

    1. **System information:**
       - Drupal version
       - PHP version
       - Module version
       - Operating system

    2. **Configuration:**
       - Server URLs
       - Environment variables
       - Docker setup (if applicable)

    3. **Error logs:**
       - Drupal logs
       - Notification server logs
       - Browser console errors
       - Redis logs

    4. **Test results:**
       - Connectivity tests
       - API responses
       - WebSocket connection status

### Support Channels

- **Issue Queue**: [Project Issues](https://www.drupal.org/project/issues/notification_server)
- **Notification Server**: [GitHub Issues](https://github.com/d34dman/notification-server/issues)
- **Community**: Drupal Slack channels

### Known Issues

Check for known issues and workarounds:

1. **Browser compatibility**: Some older browsers may have WebSocket limitations
2. **Proxy servers**: Corporate firewalls may block WebSocket connections
3. **Container networking**: Docker network configuration may require adjustment
4. **Performance limits**: Default configuration is optimized for development, not production
