# Dependencies

This page covers the setup and configuration of external dependencies required by the Notification Server module.

!!! warning "Required Dependencies"
    This module is a **client** for the external [Notification Server](https://github.com/d34dman/notification-server) and cannot function without it. The notification server requires Redis for data persistence and storage.

## External Notification Server

The notification server provides:

- **WebSocket server** for real-time communication
- **HTTP API** for publishing notifications
- **Channel and client management**
- **Access control and security features**

## Setup Options

### Option A: DDEV (Recommended for Development)

For Drupal developers using DDEV, the notification server can be easily added using the DDEV addon:

```bash
# Add the notification server to your DDEV project
ddev add-on get d34dman/ddev-notification-server
ddev restart
```

**Benefits:**
- Automatic Redis setup
- Pre-configured networking
- Easy integration with existing DDEV projects
- No manual configuration required

More information: [DDEV Notification Server Addon](https://addons.ddev.com/addons/d34dman/ddev-notification-server)

!!! tip "Support DDEV"
    Please consider [supporting DDEV](https://ddev.com/support-ddev/) if you appreciate the service.

### Option B: Docker Compose (Recommended for Production)

Create a `docker-compose.yml` file:

```yaml
version: "3.8"

services:
  notification-server:
    image: ghcr.io/d34dman/notification-server:main
    ports:
      - "3000:3000"  # HTTP API
      - "8080:8080"  # WebSocket for real-time notifications
    environment:
      - REDIS_URL=redis://redis:6379
      - WS_PORT=8080
      - CORS_ORIGIN=*
      - PORT=3000
    networks:
      - notification-network
    depends_on:
      - redis

  redis:
    image: redis:alpine
    ports:
      - "6379:6379"
    networks:
      - notification-network

networks:
  notification-network:
    driver: bridge
```

Then run:
```bash
docker-compose up -d
```

**Benefits:**
- Production-ready setup
- Automatic Redis integration
- Easy scaling and management
- Isolated network environment

### Option C: Docker

Simple Docker setup without compose:

```bash
# Start Redis
docker run -d --name redis -p 6379:6379 redis:alpine

# Start notification server (link to Redis)
docker run -d --name notification-server \
  --link redis:redis \
  -p 3000:3000 -p 8080:8080 \
  -e REDIS_URL=redis://redis:6379 \
  ghcr.io/d34dman/notification-server:main
```

### Option D: Manual Installation

If you prefer to install and run components manually:

#### 1. Install and Setup Redis

**macOS with Homebrew:**
```bash
brew install redis
brew services start redis
```

**Ubuntu/Debian:**
```bash
sudo apt-get update
sudo apt-get install redis-server
sudo systemctl start redis-server
sudo systemctl enable redis-server
```

**CentOS/RHEL:**
```bash
sudo yum install epel-release
sudo yum install redis
sudo systemctl start redis
sudo systemctl enable redis
```

**Docker (Alternative):**
```bash
docker run -d -p 6379:6379 --name redis redis:alpine
```

#### 2. Install and Setup Notification Server

```bash
# Clone the notification server repository
git clone https://github.com/d34dman/notification-server.git
cd notification-server

# Install Node.js dependencies
npm install

# Configure environment
cp .env.example .env
```

Edit the `.env` file with your configuration:
```env
# Server configuration
PORT=3000
WS_PORT=8080

# Redis configuration
REDIS_URL=redis://localhost:6379

# CORS configuration (adjust for production)
CORS_ORIGIN=*

# Optional: Authentication settings
# JWT_SECRET=your-secret-key
```

Start the server:
```bash
# Development mode
npm run dev

# Production mode
npm start
```

**Benefits:**
- Full control over configuration
- Ability to modify source code
- Custom deployment options
- Integration with existing infrastructure

## Configuration

### Drupal Module Configuration

After setting up the notification server, configure the Drupal module:

1. **Navigate to configuration**: Visit `/admin/config/system/notification-server`

2. **Configure server URLs**:

   **For DDEV:**
   - **Server URL**: `http://notification-server:3000`
   - **WebSocket URL**: `ws://notification-server:8080`

   **For Docker/Docker Compose (local):**
   - **Server URL**: `http://localhost:3000`
   - **WebSocket URL**: `ws://localhost:8080`

   !!! note "Port Configuration"
       The ports 3000 (HTTP) and 8080 (WebSocket) are the default ports used by the notification server. The original settings form examples may show different ports (like 3111) but the actual notification server defaults to 3000 for HTTP.

   **For production:**
   - **Server URL**: `https://your-notification-server.com:3000`
   - **WebSocket URL**: `wss://your-notification-server.com:8080`

3. **Save configuration**

### Environment Variables

The notification server supports various environment variables:

| Variable | Default | Description |
|----------|---------|-------------|
| `PORT` | 3000 | HTTP API port |
| `WS_PORT` | 8080 | WebSocket port |
| `REDIS_URL` | `redis://localhost:6379` | Redis connection URL |
| `CORS_ORIGIN` | `*` | CORS allowed origins |
| `JWT_SECRET` | (none) | JWT secret for authentication |
| `LOG_LEVEL` | `info` | Logging level |
| `MAX_CONNECTIONS` | 1000 | Maximum WebSocket connections |
| `MESSAGE_TTL` | 3600 | Message time-to-live in seconds |

### Redis Configuration

For production environments, consider Redis configuration optimization:

**Memory Management:**
```redis
# Set memory policy
maxmemory 256mb
maxmemory-policy allkeys-lru
```

**Persistence:**
```redis
# Enable AOF persistence
appendonly yes
appendfsync everysec
```

**Performance:**
```redis
# Optimize for performance
tcp-keepalive 300
timeout 0
```

## Security Considerations

### Production Security

**TLS/SSL Configuration:**
- Use `wss://` for WebSocket connections in production
- Use `https://` for HTTP API calls
- Implement proper certificate management

**Network Security:**
- Restrict Redis access to notification server only
- Use firewall rules to limit port access
- Consider VPN for server-to-server communication

**Application Security:**
- Set strong `JWT_SECRET` if using authentication
- Implement rate limiting
- Monitor for suspicious activity

### Authentication Setup

For secured environments, configure JWT authentication:

```env
JWT_SECRET=your-very-secure-secret-key-here
AUTH_ENABLED=true
TOKEN_EXPIRY=3600
```

## Monitoring and Health Checks

### Health Check Endpoints

The notification server provides health check endpoints:

- **HTTP Health Check**: `GET /health`
- **WebSocket Status**: `GET /ws/status`
- **Redis Connection**: `GET /health/redis`

### Monitoring Setup

**Docker Health Checks:**
```yaml
notification-server:
  image: ghcr.io/d34dman/notification-server:main
  healthcheck:
    test: ["CMD", "curl", "-f", "http://localhost:3000/health"]
    interval: 30s
    timeout: 10s
    retries: 3
```

**Log Monitoring:**
```bash
# View notification server logs
docker logs -f notification-server

# View Redis logs
docker logs -f redis
```

## Scaling Considerations

### Horizontal Scaling

For high-traffic applications:

1. **Multiple notification server instances**
2. **Redis clustering** for data distribution
3. **Load balancer** for WebSocket connections
4. **Session affinity** for WebSocket persistence

### Performance Optimization

**Redis Optimization:**
- Use Redis Cluster for large datasets
- Implement connection pooling
- Monitor memory usage and performance

**Notification Server Optimization:**
- Increase `MAX_CONNECTIONS` based on server capacity
- Optimize `MESSAGE_TTL` for your use case
- Monitor CPU and memory usage

## Troubleshooting Dependencies

### Common Issues

**Redis Connection Issues:**
```bash
# Test Redis connectivity
redis-cli ping

# Check Redis logs
docker logs redis
```

**Notification Server Issues:**
```bash
# Check server logs
docker logs notification-server

# Test HTTP API
curl http://localhost:3000/health

# Test WebSocket (using wscat)
npm install -g wscat
wscat -c ws://localhost:8080
```

**Network Issues:**
```bash
# Test port accessibility
telnet localhost 3000
telnet localhost 8080
telnet localhost 6379
```

### Debugging Steps

1. **Verify all services are running**
2. **Check port accessibility**
3. **Validate environment variables**
4. **Review application logs**
5. **Test connectivity between components**

## Backup and Recovery

### Redis Backup

**Automated Backup:**
```bash
# Create Redis backup
redis-cli BGSAVE

# Copy backup file
cp /var/lib/redis/dump.rdb /backup/location/
```

**Recovery:**
```bash
# Stop Redis
sudo systemctl stop redis

# Restore backup
cp /backup/location/dump.rdb /var/lib/redis/

# Start Redis
sudo systemctl start redis
```

### Configuration Backup

Backup important configuration files:
- Notification server `.env` file
- Redis configuration
- Docker Compose files
- Drupal module configuration

## Version Compatibility

### Supported Versions

| Component | Minimum | Recommended |
|-----------|---------|-------------|
| Notification Server | 1.0.0 | Latest |
| Redis | 6.0 | 7.0+ |
| Node.js | 16.0 | 18.0+ |
| Docker | 20.0 | Latest |
| Docker Compose | 1.29 | Latest |

### Update Procedures

**Notification Server Updates:**
```bash
# Pull latest image
docker pull ghcr.io/d34dman/notification-server:main

# Restart with new image
docker-compose down
docker-compose up -d
```

**Redis Updates:**
```bash
# Backup data first
redis-cli BGSAVE

# Update Redis image
docker pull redis:alpine
docker-compose up -d redis
```
