# Features

The Notification Server module provides comprehensive real-time notification capabilities through integration with an external notification server.

## Core Features

### HTTP Notifications

- **Channel-based message publishing** - Send messages to specific channels
- **Channel creation and management** - Create channels with custom access rules
- **Client ID generation and validation** - Manage WebSocket client identities
- **Notification history retrieval** - Get previous messages from channels

### WebSocket Support

- **Real-time bidirectional communication** - Instant message delivery to connected clients
- **Channel subscription management** - Subscribe/unsubscribe from channels dynamically
- **Message broadcasting to specific channels** - Send messages to all channel subscribers
- **Client connection management** - Handle client authentication and access control

### Administration

- **Configuration interface for server settings** - Easy setup through Drupal admin interface
- **Permission-based access control** - Control who can configure and use the module
- **Demo interfaces for testing functionality** - Built-in demos for HTTP and WebSocket features

## Channel Management

### Channel Creation

Create channels with flexible access control rules:

!!! example "Creating a Channel with Access Rules"

    ```php
    <?php
    use Drupal\notification_server\DTO\ChannelDTO;
    use Drupal\notification_server\DTO\ChannelRulesDTO;

    $rules = new ChannelRulesDTO(
      allowedClientIds: [$client_id],
      isPublic: false,
      allowedPatterns: ['client.*'],
      maxSubscribers: 100
    );

    $channel = new ChannelDTO('my_channel', $rules);
    $notification_client->createChannel($channel);
    ```

### Access Control

- **Client-specific access** - Grant/revoke access to specific client IDs
- **Pattern-based access** - Use regex patterns for client ID matching
- **Public channels** - Allow unrestricted access when needed
- **Subscriber limits** - Control maximum number of concurrent subscribers

### Channel Rules

Configure channels with:

- `allowedClientIds` - Array of specific client IDs allowed to access the channel
- `isPublic` - Boolean indicating if the channel is publicly accessible
- `allowedPatterns` - Array of regex patterns for matching allowed client IDs
- `maxSubscribers` - Maximum number of subscribers (0 for unlimited)

## Client Management

### Client ID Generation

Generate unique client IDs for WebSocket connections:

!!! example "Client ID Management"

    ```php
    <?php
    // Generate with metadata
    $client_id = $notification_client->generateClientId(null, [
      'userAgent' => 'Drupal Client',
      'environment' => 'production',
      'userId' => $user->id()
    ]);

    // Validate existing client ID
    $is_valid = $notification_client->validateClientId($client_id);

    // Remove client and all associated data
    $notification_client->removeClient($client_id);
    ```

### Client Features

- **Automatic client ID generation** - Create unique identifiers for WebSocket connections
- **Client validation** - Verify client ID validity before connections
- **Metadata support** - Store additional information with client IDs
- **Client cleanup** - Remove clients and associated data when no longer needed

## Notification Publishing

### Message Types

Support for various message formats:

!!! example "Different Message Types"

    ```php
    <?php
    // Simple string message
    $notification_client->publishNotification('channel', 'Hello World');

    // Structured data
    $notification_client->publishNotification('updates', [
      'type' => 'content_update',
      'message' => 'New article published',
      'data' => [
        'nodeId' => 123,
        'title' => 'Breaking News',
        'author' => 'John Doe'
      ]
    ]);

    // Complex objects
    $notification_client->publishNotification('events', $event_object);
    ```

### Message History

Retrieve notification history:

!!! example "Getting Message History"

    ```php
    <?php
    // Get last 10 notifications
    $notifications = $notification_client->getNotifications('channel_name');

    // Get specific number of notifications
    $notifications = $notification_client->getNotifications('channel_name', 25);
    ```

## WebSocket Integration

### Real-time Communication

- **Instant message delivery** - Messages delivered immediately to connected clients
- **Bidirectional communication** - Clients can send and receive messages
- **Connection status tracking** - Monitor client connection states
- **Automatic reconnection support** - Handle connection drops gracefully

### JavaScript Integration

!!! example "WebSocket Client Setup"

    ```javascript
    // Establish WebSocket connection
    const websocketUrl = drupalSettings.notificationServer.websocketUrl;
    const socket = new WebSocket(`${websocketUrl}?clientId=${clientId}`);

    // Handle connection events
    socket.onopen = () => console.log('Connected');
    socket.onclose = () => console.log('Disconnected');
    socket.onerror = (error) => console.error('WebSocket error:', error);

    // Handle incoming messages
    socket.onmessage = (event) => {
      const data = JSON.parse(event.data);
      if (data.type === 'notification') {
        // Process notification
        handleNotification(data.data);
      }
    };

    // Subscribe to channels
    socket.send(JSON.stringify({
      type: 'subscription',
      action: 'subscribe',
      channel: 'my_channel'
    }));
    ```

## Demo Module Features

The optional `notification_server_demo` module provides:

### HTTP Demo Interface

- **Message publishing form** - Test sending notifications via HTTP
- **Channel selection** - Choose target channels for messages
- **Message formatting** - Test different message types and structures
- **Response display** - View server responses and success/error states

### WebSocket Demo Interface

- **Client connection management** - Connect/disconnect WebSocket clients
- **Channel subscription** - Subscribe/unsubscribe from channels dynamically
- **Real-time message display** - View incoming notifications in real-time
- **Connection status monitoring** - Track WebSocket connection state
- **Message log** - History of all received messages

## Security Features

!!! warning "Security Considerations"
    
    Always implement proper security measures when using the notification server in production environments. Never expose sensitive data through public channels.

### Access Control

- **Permission-based administration** - Control who can configure the module
- **Channel-level access control** - Restrict access to specific channels
- **Client authentication** - Validate client IDs before allowing connections
- **Pattern-based restrictions** - Use regex patterns for flexible access rules

### Data Validation

- **Input sanitization** - Validate all incoming data
- **Message size limits** - Prevent oversized message payloads
- **Rate limiting** - Control message frequency (server-side)
- **Connection limits** - Restrict maximum concurrent connections

## Performance Features

!!! tip "Performance Optimization"

    For optimal performance in production, consider configuring Redis memory limits, implementing connection pooling, and monitoring message throughput patterns.

### Efficient Communication

- **WebSocket persistence** - Maintain long-lived connections for real-time updates
- **Message batching** - Group multiple notifications when appropriate
- **Connection pooling** - Reuse connections efficiently
- **Minimal overhead** - Lightweight protocol implementation

### Scalability

- **Redis-backed storage** - Fast data persistence and retrieval
- **Horizontal scaling** - Support multiple notification server instances
- **Load balancing** - Distribute connections across servers
- **Connection management** - Handle thousands of concurrent connections

