<?php

namespace Drupal\notification_server_demo\Form;

use Drupal\Core\Ajax\PrependCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\notification_server\Service\NotificationServerClientInterface;
use Drupal\notification_server\DTO\ChannelDTO;
use Drupal\notification_server\DTO\ChannelRulesDTO;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Provides a form for testing the notification server HTTP API.
 */
final class NotificationHttpDemoForm extends FormBase {

  /**
   * Maximum number of log entries to display.
   */
  const MAX_LOG_ENTRIES = 100;

  public function __construct(
    protected readonly NotificationServerClientInterface $notificationClient,
    protected readonly LoggerChannelInterface $logger,
    protected readonly TimeInterface $time,
    protected readonly DateFormatterInterface $dateFormatter,
    protected readonly RendererInterface $renderer,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('notification_server.client'),
      $container->get('logger.factory')->get('notification_server_demo'),
      $container->get('datetime.time'),
      $container->get('date.formatter'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'notification_http_demo_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'notification_server_demo/http_demo';
    $form['#attributes']['class'][] = 'demo-form';

    $form['description'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => $this->t('Test the notification server HTTP API'),
      '#attributes' => [
        'class' => ['demo-form__description'],
      ],
    ];

    // Client Management Section.
    $form['client_management'] = [
      '#type' => 'details',
      '#title' => $this->t('Client Management'),
      '#open' => TRUE,
      '#attributes' => [
        'class' => ['demo-form__section'],
      ],
    ];

    $form['client_management']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#attributes' => [
        'class' => ['demo-form__input'],
        'placeholder' => $this->t('Enter client ID or generate a new one'),
      ],
      '#prefix' => '<div class="demo-form__field">',
      '#suffix' => '</div>',
    ];

    $form['client_management']['actions'] = [
      '#type' => 'actions',
      '#attributes' => [
        'class' => ['demo-form__actions'],
      ],
    ];

    $form['client_management']['actions']['generate'] = [
      '#type' => 'button',
      '#value' => $this->t('Generate Client ID'),
      '#ajax' => [
        'callback' => '::generateClientId',
        'progress' => ['type' => 'throbber'],
        'wrapper' => 'notification-log',
      ],
      '#attributes' => [
        'class' => ['demo-form__button', 'demo-form__button--primary'],
      ],
    ];

    $form['client_management']['actions']['validate'] = [
      '#type' => 'button',
      '#value' => $this->t('Validate Client ID'),
      '#ajax' => [
        'callback' => '::validateClientId',
        'progress' => ['type' => 'throbber'],
        'wrapper' => 'notification-log',
      ],
      '#attributes' => [
        'class' => ['demo-form__button'],
      ],
    ];

    // Channel Management Section.
    $form['channel_management'] = [
      '#type' => 'details',
      '#title' => $this->t('Channel Management'),
      '#open' => FALSE,
      '#attributes' => [
        'class' => ['demo-form__section'],
      ],
    ];

    $form['channel_management']['channel'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel'),
      '#required' => TRUE,
      '#default_value' => 'demo-channel',
      '#attributes' => [
        'class' => ['demo-form__input'],
        'placeholder' => $this->t('Enter channel name'),
      ],
      '#prefix' => '<div class="demo-form__field">',
      '#suffix' => '</div>',
    ];

    $form['channel_management']['actions'] = [
      '#type' => 'actions',
      '#attributes' => [
        'class' => ['demo-form__actions'],
      ],
    ];

    $form['channel_management']['actions']['create'] = [
      '#type' => 'button',
      '#value' => $this->t('Create Channel'),
      '#ajax' => [
        'callback' => '::createChannel',
        'progress' => ['type' => 'throbber'],
      ],
      '#attributes' => [
        'class' => ['demo-form__button', 'demo-form__button--primary'],
      ],
    ];

    $form['channel_management']['actions']['grant'] = [
      '#type' => 'button',
      '#value' => $this->t('Grant Access'),
      '#ajax' => [
        'callback' => '::grantAccess',
        'progress' => ['type' => 'throbber'],
      ],
      '#attributes' => [
        'class' => ['demo-form__button'],
      ],
    ];

    $form['channel_management']['actions']['revoke'] = [
      '#type' => 'button',
      '#value' => $this->t('Revoke Access'),
      '#ajax' => [
        'callback' => '::revokeAccess',
        'progress' => ['type' => 'throbber'],
      ],
      '#attributes' => [
        'class' => ['demo-form__button'],
      ],
    ];

    // Notification Management Section.
    $form['notification_management'] = [
      '#type' => 'details',
      '#title' => $this->t('Notification Management'),
      '#open' => FALSE,
      '#attributes' => [
        'class' => ['demo-form__section'],
      ],
    ];

    $form['notification_management']['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#required' => TRUE,
      '#default_value' => 'Hello World!',
      '#attributes' => [
        'class' => ['demo-form__textarea'],
        'placeholder' => $this->t('Enter your message'),
      ],
      '#prefix' => '<div class="demo-form__field">',
      '#suffix' => '</div>',
    ];

    $form['notification_management']['actions'] = [
      '#type' => 'actions',
      '#attributes' => [
        'class' => ['demo-form__actions'],
      ],
    ];

    $form['notification_management']['actions']['send'] = [
      '#type' => 'button',
      '#value' => $this->t('Send Notification'),
      '#ajax' => [
        'callback' => '::sendNotification',
        'progress' => ['type' => 'throbber'],
      ],
      '#attributes' => [
        'class' => ['demo-form__button', 'demo-form__button--primary'],
      ],
    ];

    $form['notification_management']['actions']['history'] = [
      '#type' => 'button',
      '#value' => $this->t('Get History'),
      '#ajax' => [
        'callback' => '::getHistory',
        'progress' => ['type' => 'throbber'],
      ],
      '#attributes' => [
        'class' => ['demo-form__button'],
      ],
    ];

    $form['log'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'notification-log',
        'class' => ['notification-log'],
      ],
    ];

    return $form;
  }

  /**
   * AJAX callback to generate a new client ID.
   */
  public function generateClientId(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    try {
      $clientId = $this->notificationClient->generateClientId();
      if ($clientId) {
        $response->addCommand(new InvokeCommand('[data-drupal-selector=edit-client-id]', 'val', [$clientId]));
        $this->addLogMessage($response, $this->t('Generated new client ID: @id', ['@id' => $clientId]));
      }
      else {
        $this->addLogMessage($response, $this->t('Failed to generate client ID'), 'error');
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error generating client ID: @error', ['@error' => $e->getMessage()]);
      $this->addLogMessage($response, $this->t('Error generating client ID. Please try again.'), 'error');
    }

    return $response;
  }

  /**
   * AJAX callback to validate a client ID.
   */
  public function validateClientId(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    // Get the client ID from the form input.
    $clientId = trim($form_state->getValue('client_id'));

    if (empty($clientId)) {
      $this->addLogMessage($response, $this->t('Please enter a client ID'), 'error');
      return $response;
    }

    try {
      $isValid = $this->notificationClient->validateClientId($clientId);
      if ($isValid) {
        $this->addLogMessage($response, $this->t('Client ID @id is valid', ['@id' => $clientId]));
      }
      else {
        $this->addLogMessage($response, $this->t('Client ID @id is invalid', ['@id' => $clientId]), 'error');
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error validating client ID: @error', ['@error' => $e->getMessage()]);
      $this->addLogMessage($response, $this->t('Error validating client ID. Please try again.'), 'error');
    }

    return $response;
  }

  /**
   * AJAX callback to create a new channel.
   */
  public function createChannel(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $channelName = $form_state->getValue('channel');

    if (empty($channelName)) {
      $this->addLogMessage($response, $this->t('Please enter a channel name'), 'error');
      return $response;
    }

    try {
      $channel = new ChannelDTO(
        channel: $channelName,
        rules: new ChannelRulesDTO(
          isPublic: FALSE,
          maxSubscribers: 100,
        ),
      );

      $success = $this->notificationClient->createChannel($channel);
      if ($success) {
        $this->addLogMessage($response, $this->t('Channel @name created successfully', ['@name' => $channelName]));
      }
      else {
        $this->addLogMessage($response, $this->t('Failed to create channel @name', ['@name' => $channelName]), 'error');
      }
    }
    catch (\Exception $e) {
      $this->addLogMessage($response, $this->t('Error: @message', ['@message' => $e->getMessage()]), 'error');
    }

    return $response;
  }

  /**
   * AJAX callback to grant channel access to a client.
   */
  public function grantAccess(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $channelName = $form_state->getValue('channel');
    $clientId = $form_state->getValue('client_id');

    if (empty($channelName) || empty($clientId)) {
      $this->addLogMessage($response, $this->t('Please enter both channel name and client ID'), 'error');
      return $response;
    }

    try {
      $success = $this->notificationClient->grantChannelAccess($channelName, $clientId);
      if ($success) {
        $this->addLogMessage($response, $this->t('Access granted to client @id for channel @name', [
          '@id' => $clientId,
          '@name' => $channelName,
        ]));
      }
      else {
        $this->addLogMessage($response, $this->t('Failed to grant access to client @id for channel @name', [
          '@id' => $clientId,
          '@name' => $channelName,
        ]), 'error');
      }
    }
    catch (\Exception $e) {
      $this->addLogMessage($response, $this->t('Error: @message', ['@message' => $e->getMessage()]), 'error');
    }

    return $response;
  }

  /**
   * AJAX callback to revoke channel access from a client.
   */
  public function revokeAccess(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $channelName = $form_state->getValue('channel');
    $clientId = $form_state->getValue('client_id');

    if (empty($channelName) || empty($clientId)) {
      $this->addLogMessage($response, $this->t('Please enter both channel name and client ID'), 'error');
      return $response;
    }

    try {
      $success = $this->notificationClient->revokeChannelAccess($channelName, $clientId);
      if ($success) {
        $this->addLogMessage($response, $this->t('Access revoked from client @id for channel @name', [
          '@id' => $clientId,
          '@name' => $channelName,
        ]));
      }
      else {
        $this->addLogMessage($response, $this->t('Failed to revoke access from client @id for channel @name', [
          '@id' => $clientId,
          '@name' => $channelName,
        ]), 'error');
      }
    }
    catch (\Exception $e) {
      $this->addLogMessage($response, $this->t('Error: @message', ['@message' => $e->getMessage()]), 'error');
    }

    return $response;
  }

  /**
   * AJAX callback to send a notification.
   */
  public function sendNotification(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $channelName = $form_state->getValue('channel');
    $message = $form_state->getValue('message');

    if (empty($channelName) || empty($message)) {
      $this->addLogMessage($response, $this->t('Please enter both channel name and message'), 'error');
      return $response;
    }

    try {
      $result = $this->notificationClient->publishNotification($channelName, $message);
      if ($result !== NULL) {
        $this->addLogMessage($response, $this->t('Notification sent to channel @name', [
          '@name' => $channelName,
        ]));
      }
      else {
        $this->addLogMessage($response, $this->t('Failed to send notification to channel @name', [
          '@name' => $channelName,
        ]), 'error');
      }
    }
    catch (\Exception $e) {
      $this->addLogMessage($response, $this->t('Error: @message', ['@message' => $e->getMessage()]), 'error');
    }

    return $response;
  }

  /**
   * AJAX callback to get notification history.
   */
  public function getHistory(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $channelName = $form_state->getValue('channel');

    if (empty($channelName)) {
      $this->addLogMessage($response, $this->t('Please enter a channel name'), 'error');
      return $response;
    }

    try {
      $notifications = $this->notificationClient->getNotifications($channelName, self::MAX_LOG_ENTRIES);
      if (!empty($notifications)) {
        foreach ($notifications as $notification) {
          $this->addLogMessage($response, $this->t('[@time] @message', [
            '@time' => $notification['timestamp'] ?? 'Unknown',
            '@message' => $notification['message'] ?? 'No message',
          ]));
        }
        $this->addLogMessage($response, (string) $this->t('Retrieved @count notification(s)', [
          '@count' => count($notifications),
        ]));
      }
      else {
        $this->addLogMessage($response, $this->t('No notifications found for channel @name', [
          '@name' => $channelName,
        ]));
      }
    }
    catch (\Exception $e) {
      $this->addLogMessage($response, $this->t('Error: @message', ['@message' => $e->getMessage()]), 'error');
    }

    return $response;
  }

  /**
   * Adds a message to the notification log.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   * @param string $message
   *   The message to add.
   * @param string $type
   *   The message type (success or error).
   */
  protected function addLogMessage(AjaxResponse $response, string $message, string $type = 'success'): void {
    $timestamp = $this->time->getCurrentTime();
    $formattedTime = $this->dateFormatter->format($timestamp, 'custom', 'H:i:s');

    $logEntry = [
      '#theme' => 'notification_log_entry',
      '#timestamp' => $formattedTime,
      '#message' => $message,
      '#type' => $type,
    ];

    $html = $this->renderer->renderInIsolation($logEntry);
    $response->addCommand(new PrependCommand('#notification-log', $html));

    // Trim old entries if needed.
    $response->addCommand(new InvokeCommand('#notification-log .notification-log__item:gt(' . (self::MAX_LOG_ENTRIES - 1) . ')', 'remove'));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Form submission is handled via AJAX callbacks.
  }

}
