<?php

declare(strict_types=1);

namespace Drupal\notification_server\DTO;

/**
 * Data Transfer Object for Channel data.
 *
 * Represents a notification channel with its associated access rules.
 */
final class ChannelDTO {

  /**
   * Constructs a new ChannelDTO object.
   *
   * @param string $channel
   *   The channel name.
   * @param \Drupal\notification_server\DTO\ChannelRulesDTO|null $rules
   *   Optional channel access rules. NULL if no specific rules are set.
   */
  public function __construct(
    private readonly string $channel,
    private readonly ?ChannelRulesDTO $rules = NULL,
  ) {}

  /**
   * Creates a ChannelDTO from an array of data.
   *
   * @param array $data
   *   The channel data array with the following structure:
   *   - channel: (string) The channel name.
   *   - rules: (array|null) Optional rules configuration.
   *
   * @return self
   *   A new ChannelDTO instance.
   */
  public static function fromArray(array $data): self {
    return new self(
      channel: $data['channel'],
      rules: isset($data['rules']) ? ChannelRulesDTO::fromArray($data['rules']) : NULL,
    );
  }

  /**
   * Gets the channel name.
   *
   * @return string
   *   The channel name.
   */
  public function getChannel(): string {
    return $this->channel;
  }

  /**
   * Gets the channel rules.
   *
   * @return \Drupal\notification_server\DTO\ChannelRulesDTO|null
   *   The channel rules configuration, or NULL if no specific rules are set.
   */
  public function getRules(): ?ChannelRulesDTO {
    return $this->rules;
  }

  /**
   * Converts the DTO to an array.
   *
   * @return array
   *   The array representation of the DTO with the following structure:
   *   - channel: (string) The channel name.
   *   - rules: (array|null) The rules configuration if set.
   *   Note: NULL values are excluded from the returned array.
   */
  public function toArray(): array {
    $data = [
      'channel' => $this->channel,
    ];

    if ($this->rules !== NULL) {
      $data['rules'] = $this->rules->toArray();
    }

    return $data;
  }

}
