<?php

declare(strict_types=1);

namespace Drupal\notification_server\DTO;

/**
 * Data Transfer Object for Channel Rules data.
 *
 * Represents access control rules for a notification channel, including client
 * access patterns, public visibility, and subscription limits.
 */
final class ChannelRulesDTO {

  /**
   * Constructs a new ChannelRulesDTO object.
   *
   * @param array|null $allowedClientIds
   *   List of specific client IDs that are allowed to access the channel.
   *   NULL if no specific client IDs are set.
   * @param bool|null $isPublic
   *   Whether the channel is public (accessible without explicit access).
   *   NULL if public status is not specified.
   * @param array|null $allowedPatterns
   *   List of patterns for matching allowed client IDs.
   *   NULL if no patterns are specified.
   * @param int|null $maxSubscribers
   *   Maximum number of subscribers allowed (0 for unlimited).
   *   NULL if no limit is specified.
   */
  public function __construct(
    private readonly ?array $allowedClientIds = NULL,
    private readonly ?bool $isPublic = NULL,
    private readonly ?array $allowedPatterns = NULL,
    private readonly ?int $maxSubscribers = NULL,
  ) {}

  /**
   * Creates a ChannelRulesDTO from an array of data.
   *
   * @param array $data
   *   The rules data array with the following structure:
   *   - allowedClientIds: (array|null) List of allowed client IDs.
   *   - isPublic: (bool|null) Whether the channel is public.
   *   - allowedPatterns: (array|null) List of allowed client ID patterns.
   *   - maxSubscribers: (int|null) Maximum number of subscribers.
   *
   * @return self
   *   A new ChannelRulesDTO instance.
   */
  public static function fromArray(array $data): self {
    return new self(
      allowedClientIds: $data['allowedClientIds'] ?? NULL,
      allowedPatterns: $data['allowedPatterns'] ?? NULL,
      maxSubscribers: $data['maxSubscribers'] ?? NULL,
      isPublic: $data['isPublic'] ?? NULL,
    );
  }

  /**
   * Gets the list of allowed client IDs.
   *
   * @return array|null
   *   Array of specific client IDs allowed to access the channel,
   *   or NULL if no specific clients are set.
   */
  public function getAllowedClientIds(): ?array {
    return $this->allowedClientIds;
  }

  /**
   * Gets whether the channel is public.
   *
   * @return bool|null
   *   TRUE if the channel is public (accessible without explicit access),
   *   FALSE if the channel requires explicit access,
   *   NULL if public status is not specified.
   */
  public function isPublic(): ?bool {
    return $this->isPublic;
  }

  /**
   * Gets the list of allowed patterns.
   *
   * @return array|null
   *   Array of patterns for matching allowed client IDs,
   *   or NULL if no patterns are specified.
   */
  public function getAllowedPatterns(): ?array {
    return $this->allowedPatterns;
  }

  /**
   * Gets the maximum number of subscribers allowed.
   *
   * @return int|null
   *   Maximum number of subscribers (0 for unlimited),
   *   or NULL if no limit is specified.
   */
  public function getMaxSubscribers(): ?int {
    return $this->maxSubscribers;
  }

  /**
   * Converts the DTO to an array.
   *
   * @return array
   *   The array representation of the DTO with the following structure:
   *   - allowedClientIds: (array|null) List of allowed client IDs.
   *   - isPublic: (bool|null) Whether the channel is public.
   *   - allowedPatterns: (array|null) List of allowed client ID patterns.
   *   - maxSubscribers: (int|null) Maximum number of subscribers.
   *   Note: NULL values are excluded from the returned array.
   */
  public function toArray(): array {
    $data = [
      'allowedClientIds' => $this->allowedClientIds,
      'allowedPatterns' => $this->allowedPatterns,
      'maxSubscribers' => $this->maxSubscribers,
      'isPublic' => $this->isPublic,
    ];

    return array_filter($data, fn($value) => $value !== NULL);
  }

}
