<?php

declare(strict_types=1);

namespace Drupal\notification_server\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Notification server settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'notification_server_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['notification_server.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['server_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Server URL'),
      '#default_value' => $this->config('notification_server.settings')->get('server_url'),
      '#description' => $this->t('The URL of the server including the port number (e.g. http://localhost:8080, http://notification-server:3111)'),
    ];
    $form['websocket_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Websocket URL'),
      '#default_value' => $this->config('notification_server.settings')->get('websocket_url'),
      '#description' => $this->t('The URL of the websocket server including the port number (e.g. ws://localhost:8080)'),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {

    $websocket_url = $form_state->getValue('websocket_url');
    if (!$this->isValidWebsocketUrl($websocket_url)) {
      $form_state->setErrorByName(
        'websocket_url',
        $this->t('Invalid websocket URL. Please provide a valid websocket URL (ws:// or wss://) with a port number.'),
      );
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('notification_server.settings')
      ->set('server_url', $form_state->getValue('server_url'))
      ->set('websocket_url', $form_state->getValue('websocket_url'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Validates the websocket URL format and port number.
   *
   * @param string $websocket_url
   *   The websocket URL to validate.
   *
   * @return bool
   *   TRUE if the URL is valid, FALSE otherwise.
   */
  private function isValidWebsocketUrl(string $websocket_url): bool {
    // Check for ws:// or wss:// protocol.
    if (!preg_match('/^wss?:\/\//i', $websocket_url)) {
      return FALSE;
    }

    // Parse the URL to check port and other components.
    $parsed_url = parse_url($websocket_url);

    // Ensure we have required components.
    if (!$parsed_url || !isset($parsed_url['host'])) {
      return FALSE;
    }

    // Check if port is specified and is a valid number.
    if (isset($parsed_url['port'])) {
      // Cast to int to ensure we have a numeric value.
      $port = (int) $parsed_url['port'];
      // Only check lower bound as upper bound is guaranteed by parse_url()
      if ($port < 1) {
        return FALSE;
      }
    }

    return TRUE;
  }

}
