<?php

declare(strict_types=1);

namespace Drupal\notification_server\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\notification_server\DTO\ChannelDTO;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\HttpFoundation\Response;

/**
 * Service for interacting with the Notification Server API.
 */
class NotificationServerClient implements NotificationServerClientInterface {

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->logger = $logger_factory->get('notification_server');
  }

  /**
   * {@inheritDoc}
   */
  protected function getServerUrl(): string {
    $url = $this->configFactory->get('notification_server.settings')->get('server_url') ?: '';
    return rtrim($url, '/');
  }

  /**
   * {@inheritDoc}
   */
  public function publishNotification(string $channel, object|array|string|float|int|bool|null $message): ?array {
    try {
      $response = $this->httpClient->request('POST', $this->getServerUrl() . '/api/notifications', [
        'json' => [
          'channel' => $channel,
          'message' => $message,
        ],
      ]);

      if ($response->getStatusCode() === Response::HTTP_OK) {
        return json_decode($response->getBody()->getContents(), TRUE);
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to publish notification: @error', ['@error' => $e->getMessage()]);
    }

    return NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getNotifications(string $channel, int $limit = 10): array {
    try {
      $response = $this->httpClient->request('GET', $this->getServerUrl() . '/api/notifications/' . $channel, [
        'query' => [
      // Ensure limit is between 1 and 100.
          'limit' => max(1, min($limit, 100)),
        ],
      ]);

      if ($response->getStatusCode() === Response::HTTP_OK) {
        return json_decode($response->getBody()->getContents(), TRUE) ?? [];
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to get notifications: @error', ['@error' => $e->getMessage()]);
    }

    return [];
  }

  /**
   * {@inheritDoc}
   */
  public function subscribeToChannel(string $channel, string $clientId): bool {
    try {
      $response = $this->httpClient->request('POST', $this->getServerUrl() . '/api/channels/' . $channel . '/subscribe', [
        'json' => [
          'clientId' => $clientId,
        ],
      ]);

      return $response->getStatusCode() === Response::HTTP_OK;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to subscribe to channel: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function unsubscribeFromChannel(string $channel, string $clientId): bool {
    try {
      $response = $this->httpClient->request('POST', $this->getServerUrl() . '/api/channels/' . $channel . '/unsubscribe', [
        'json' => [
          'clientId' => $clientId,
        ],
      ]);

      return $response->getStatusCode() === Response::HTTP_OK;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to unsubscribe from channel: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function generateClientId(?string $client_id = NULL, array $metadata = []): ?string {
    try {
      $response = $this->httpClient->request('POST', $this->getServerUrl() . '/api/clients', [
        'json' => [
          'clientId' => $client_id,
          'metadata' => $metadata,
        ],
      ]);

      if ($response->getStatusCode() === Response::HTTP_OK || $response->getStatusCode() === Response::HTTP_CREATED) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return $data['clientId'] ?? NULL;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to generate client ID: @error', ['@error' => $e->getMessage()]);
    }

    return NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function removeClient(string $clientId): bool {
    try {
      $response = $this->httpClient->request('DELETE', $this->getServerUrl() . "/api/clients/{$clientId}");
      if ($response->getStatusCode() === Response::HTTP_OK) {
        return TRUE;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to remove client ID: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function validateClientId(string $clientId): bool {
    try {
      $response = $this->httpClient->request('GET', $this->getServerUrl() . "/api/clients/{$clientId}");

      if ($response->getStatusCode() === Response::HTTP_OK) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return $data['isValid'] ?? FALSE;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to validate client ID: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function createChannel(ChannelDTO $channel): bool {
    try {
      $response = $this->httpClient->request('POST', $this->getServerUrl() . '/api/channels', [
        'json' => $channel->toArray(),
      ]);

      if ($response->getStatusCode() === Response::HTTP_CREATED) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return isset($data['channel']);
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to create channel: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function grantChannelAccess(string $channel, string $clientId): bool {
    try {
      $response = $this->httpClient->request('POST', $this->getServerUrl() . "/api/channels/{$channel}/access/{$clientId}");

      if ($response->getStatusCode() === Response::HTTP_OK) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return $data['accessGranted'] ?? FALSE;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to grant channel access: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function revokeChannelAccess(string $channel, string $clientId): bool {
    try {
      $response = $this->httpClient->request('DELETE', $this->getServerUrl() . "/api/channels/{$channel}/access/{$clientId}");

      if ($response->getStatusCode() === Response::HTTP_OK) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return $data['accessRevoked'] ?? FALSE;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to revoke channel access: @error', ['@error' => $e->getMessage()]);
    }
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function getWebsocketEndpoint(): string {
    $config = $this->configFactory->get('notification_server.settings');
    return $config->get('websocket_url');
  }

}
