<?php

namespace Drupal\notification_server\Service;

use Drupal\notification_server\DTO\ChannelDTO;

/**
 * Service for interacting with the Notification Server API.
 */
interface NotificationServerClientInterface {

  /**
   * Generate a new client ID.
   *
   * @param string|null $clientId
   *   The client ID to validate/generate.
   * @param array $metadata
   *   Optional metadata for the client (e.g., userAgent, environment).
   *
   * @return string|null
   *   The generated client ID if successful, NULL otherwise.
   */
  public function generateClientId(?string $clientId = NULL, array $metadata = []): ?string;

  /**
   * Delete the client and all its data.
   *
   * @param string $clientId
   *   The client ID to delete.
   *
   * @return bool
   *   True if successful.
   */
  public function removeClient(string $clientId): bool;

  /**
   * Validate a client ID.
   *
   * @param string $clientId
   *   The client ID to validate.
   *
   * @return bool
   *   TRUE if the client ID is valid, FALSE otherwise.
   */
  public function validateClientId(string $clientId): bool;

  /**
   * Create a new channel.
   *
   * @param \Drupal\notification_server\DTO\ChannelDTO $channel
   *   The channel data.
   *
   * @return bool
   *   TRUE if channel was created successfully, FALSE otherwise.
   */
  public function createChannel(ChannelDTO $channel): bool;

  /**
   * Grant access to a channel for a client.
   *
   * @param string $channel
   *   The channel name.
   * @param string $clientId
   *   The client ID to grant access to.
   *
   * @return bool
   *   TRUE if access was granted successfully, FALSE otherwise.
   */
  public function grantChannelAccess(string $channel, string $clientId): bool;

  /**
   * Revoke access to a channel for a client.
   *
   * @param string $channel
   *   The channel name.
   * @param string $clientId
   *   The client ID to revoke access from.
   *
   * @return bool
   *   TRUE if access was revoked successfully, FALSE otherwise.
   */
  public function revokeChannelAccess(string $channel, string $clientId): bool;

  /**
   * Publish a new notification.
   *
   * @param string $channel
   *   The channel to publish the notification to.
   * @param object|array|string|float|int|bool|null $message
   *   The notification message content.
   *
   * @return array|null
   *   The notification data if successful, NULL otherwise.
   */
  public function publishNotification(string $channel, object|array|string|float|int|bool|null $message): ?array;

  /**
   * Get notification history for a channel.
   *
   * @param string $channel
   *   The channel name.
   * @param int $limit
   *   Maximum number of notifications to return (default: 10).
   *
   * @return array
   *   Array of notifications.
   */
  public function getNotifications(string $channel, int $limit = 10): array;

  /**
   * Subscribe a client to a channel.
   *
   * @param string $channel
   *   The channel name to subscribe to.
   * @param string $clientId
   *   The WebSocket client ID.
   *
   * @return bool
   *   TRUE if subscription was successful, FALSE otherwise.
   */
  public function subscribeToChannel(string $channel, string $clientId): bool;

  /**
   * Unsubscribe a client from a channel.
   *
   * @param string $channel
   *   The channel name to unsubscribe from.
   * @param string $clientId
   *   The WebSocket client ID.
   *
   * @return bool
   *   TRUE if unsubscription was successful, FALSE otherwise.
   */
  public function unsubscribeFromChannel(string $channel, string $clientId): bool;

  /**
   * Gets the WebSocket server URL.
   *
   * @return string
   *   The WebSocket server URL.
   */
  public function getWebsocketEndpoint(): string;

}
