<?php

namespace Drupal\notify_manager\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Link;
use Drupal\user\UserStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\notify_manager\NotificationService;

/**
 * Class Notification Controller.
 */
class NotificationController extends ControllerBase {

  /**
   * Notify Manager.
   *
   * @var \Drupal\notify_manager\NotificationService
   */
  protected $notifyService;

  /**
   * Database connection variable.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Notification Type Variable.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $managerType;

  /**
   * Account Interface.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * Construct of Notification Controller object.
   */
  public function __construct(NotificationService $notify_service, Connection $connection, EntityStorageInterface $storage, UserStorageInterface $user) {
    $this->notifyService = $notify_service;
    $this->database = $connection;
    $this->managerType = $storage;
    $this->account = $user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('notify.service'),
      $container->get('database'),
      $container->get('entity_type.manager')->getStorage('notify_manager_type'),
      $container->get('entity_type.manager')->getStorage('user'),
    );
  }

  /**
   * Allows all notifications of the current user.
   */
  public function allNotifications() {
    $header = [
      'id' => [
        'data' => $this->t('ID'),
      ],
      'title' => [
        'data' => $this->t('Title'),
      ],
      'description' => [
        'data' => $this->t('Description'),
      ],
      'view' => [
        'data' => $this->t('Operations'),
      ],
    ];
    $rows = [];
    $notifications = $this->notifyService->notificationList(0, 10, 'total', TRUE);
    if (!empty($notifications)) {
      foreach ($notifications as $notification) {
        $rows[] = [
          'id' => $notification->id(),
          'title' => $notification->label(),
          'description' => substr(strip_tags($notification->description->value), 0, 100) . '...',
          'view' => Link::createFromRoute('View', 'entity.notify_manager.canonical', ['notify_manager' => $notification->id()]),
        ];
      }
    }

    if (!empty($rows)) {
      foreach ($rows as $key => $row) {
        $rows[$key] = [
          'data' => $row,
          'class' => [$this->notifyService->statusCheck($row['id'])],
        ];
      }
    }

    $build['table'] = [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No notifications has been found.'),
      '#attached' => [
        'library' => 'notify_manager/notify_css',
      ],
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];
    return $build;
  }

  /**
   * Returns notifications of the specific notification ID.
   */
  public function notificationId($id = 0) {
    $query = $this->database->select('notify_status', 'ns')
      ->fields('ns', ['id', 'uid', 'bundle'])
      ->condition('ns.entity_id', $id);
    $pager = $query->extend(PagerSelectExtender::class)->limit(10);
    $result = $pager->execute()->fetchAll();
    $header = [
      'id' => [
        'data' => $this->t('ID'),
      ],
      'bundle' => [
        'data' => $this->t('Type'),
      ],
      'uid' => [
        'data' => $this->t('UID'),
      ],
      'email' => [
        'data' => $this->t('Email'),
      ],
    ];
    $rows = [];
    foreach ($result as $value) {
      $user = $this->account->load($value->uid);
      $rows[] = [
        'id' => $value->id,
        'bundle' => $this->managerType->load($value->bundle)->label(),
        'uid' => $user->id(),
        'email' => $user->mail->value,
      ];
    }
    $build['table'] = [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No users has seen the notification.'),
      '#attached' => [
        'library' => 'notify_manager/notify_css',
      ],
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];
    return $build;
  }

}
