<?php

namespace Drupal\notify_manager;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Notification Service.
 */
class NotificationService {

  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * ManagerType.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $notifyManager;

  /**
   * User Storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * Account Interface.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * Database connection variable.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Construct method.
   */
  public function __construct(EntityTypeManagerInterface $manager, AccountProxyInterface $account, Connection $connection) {
    $this->entityTypeManager = $manager;
    $this->notifyManager = $manager->getStorage('notify_manager');
    $this->userStorage = $manager->getStorage('user');
    $this->account = $account;
    $this->database = $connection;
  }

  /**
   * Total count of all notifications.
   */
  public function totalCount() {
    return $this->notifyManager->getQuery()->condition('status', 1)
      ->count()->execute();
  }

  /**
   * Total count of current user notifications.
   */
  public function userCount() {
    $notifyQuery = $this->notifyQuery(0, 0, 'total');
    if (empty($notifyQuery)) {
      return 0;
    }
    return $notifyQuery->count()->execute();
  }

  /**
   * List of notifications.
   */
  public function notificationList($offset = 0, $limit = 5, $type = 'list', $pager = FALSE) {
    $notifyQuery = $this->notifyQuery($offset, $limit, $type);
    if ($pager) {
      $notifyQuery->pager($limit);
    }
    $notifiyResults = $notifyQuery->execute();
    return $this->notifyManager->loadMultiple($notifiyResults);
  }

  /**
   * To check the status of the notification.
   */
  public function statusCheck($entityId = 0) {
    $result = $this->database->select('notify_status', 'ns')
      ->fields('ns', ['id'])
      ->condition('ns.entity_id', $entityId)
      ->condition('ns.uid', $this->account->id())
      ->execute()->fetchField();
    $value = 'unread';
    if ($result) {
      $value = 'read';
    }
    return $value;
  }

  /**
   * Preparing the common query for all the notifications.
   */
  private function notifyQuery($offset = 0, $limit = 5, $type = 'total') {
    $notifyQuery = $this->notifyManager->getQuery()->condition('status', 1);

    $user = $this->userStorage->load($this->account->id());

    $userGroups = [];
    if (!empty($user->field_notification_groups->target_id)) {
      foreach ($user->field_notification_groups->getValue() as $val) {
        $userGroups[] = $val['target_id'];
      }
    }
    $orCon = $notifyQuery->orConditionGroup();
    $orCon->condition('field_notification_mode', 'global');
    if (!empty($userGroups)) {
      $orCon->condition('field_notification_groups', $userGroups, 'IN');
    }
    $orCon->condition('field_notification_users', [$this->account->id()], 'IN');
    $notifyQuery->condition($orCon);
    if ($type !== 'total') {
      $notifyQuery->range($offset, $limit);
    }
    return $notifyQuery;
  }

}
