<?php

namespace Drupal\notify_manager\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\notify_manager\NotificationService;
use Drupal\user\UserStorageInterface;

/**
 * Provides an Notification block.
 *
 * @Block(
 *   id = "notification_block",
 *   admin_label = @Translation("Notification Block"),
 *   category = @Translation("Notifications")
 * )
 */
class NotificationsBlock extends BlockBase implements ContainerFactoryPluginInterface {
  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Account Interface.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * User Storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * Notify Manager.
   *
   * @var \Drupal\notify_manager\NotificationService
   */
  protected $notifyManager;

  /**
   * Construct of Notification Block object.
   *
   * @param array $configuration
   *   Configuration variable.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin Definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $manager
   *   The entity type definition.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The account definition.
   * @param \Drupal\user\UserStorageInterface $user_storage
   *   The user storage definition.
   * @param \Drupal\notify_manager\NotificationService $notify_manager
   *   The notification definition.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $manager, AccountProxyInterface $account, UserStorageInterface $user_storage, NotificationService $notify_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $manager;
    $this->account = $account;
    $this->userStorage = $user_storage;
    $this->notifyManager = $notify_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('entity_type.manager')->getStorage('user'),
      $container->get('notify.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $notifications = $this->notifyManager->notificationList();
    $content = [];
    if (!empty($notifications)) {
      foreach ($notifications as $notify) {
        $content[] = [
          'id' => $notify->id(),
          'bundle' => $notify->bundle(),
          'title' => $notify->label(),
          'description' => $notify->description->value,
          'class' => $this->notifyManager->statusCheck($notify->id()),
        ];
      }
    }
    return [
      '#theme' => 'notify_block',
      '#content' => $content,
      '#attached' => [
        'library' => ['notify_manager/notify_css'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
