<?php

namespace Drupal\notify_widget;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\UserData;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Defines a service for comment #lazy_builder callbacks.
 */
class NotifyWidgetLazyBuilders implements TrustedCallbackInterface {

  use StringTranslationTrait;

  /**
   * The URL base path.
   *
   * @var string
   */
  protected string $basePath;

  /**
   * Constructs a new NotifyWidgetLazyBuilders object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database service.
   * @param \Drupal\notify_widget\NotifyWidgetApi $notifyWidgetApi
   *   The notify widget API service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Component\DateTime\TimeInterface $time
   *   The time service.
   * @param \Drupal\user\UserData $userData
   *   The user data service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *  The request stack service for accessing the current request.
   */
  public function __construct(
    protected readonly Connection $database,
    protected readonly NotifyWidgetApi $notifyWidgetApi,
    protected readonly AccountProxyInterface $currentUser,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly TimeInterface $time,
    protected readonly UserData $userData,
    protected readonly RequestStack $request_stack
  ) {
    $current_request = $request_stack->getCurrentRequest();
    $this->basePath = $current_request ? $current_request->getBasePath() : '';
  }

  /**
   * Lazy builder callback; builds the Notify Widget block.
   *
   * @return array
   *   A render array containing the block content.
   */
  public function renderNotifyWidgetBlock(): array {
    // Get the config settings.
    $includeRead = (bool) $this->configFactory->get('notify_widget.settings')->get('include_read') ?? TRUE;
    $readCutoff = (int) $this->configFactory->get('notify_widget.settings')->get('read_cutoff') ?? 0;

    // If readCutoff is not zero, take the current timestamp and subtract
    // the selected cutoff from it.
    if ($readCutoff !== 0) {
      $readCutoff = $this->time->getRequestTime() - $readCutoff;
    }

    // Get the notification details for the user.
    $notifications = $this->notifyWidgetApi->getNotificationsForUser(0, 0, $includeRead, $readCutoff);
    $unreadCount = $this->notifyWidgetApi->getUnreadNotificationsCount();

    // If unread count is more than 20, set the count to 20+.
    if ($unreadCount > 20) {
      $unreadCount = '20+';
    }

    // Convert the timestamp for each notification to 'time ago'.
    foreach ($notifications as $notification) {
      $notification->timestamp = $this->relativeTime($notification->timestamp);
    }

    // Get the current user's ID and the flag indicating whether they are
    // new to the notification's system.
    $uid = $this->currentUser->id();
    $newToNotifications = empty($this->userData->get('notify_widget', $uid, 'new_to_notifications'))
      ? 'yes' :
      $this->userData->get('notify_widget', $uid, 'new_to_notifications');

    // Build the render array for the block.
    $build = [];
    $build['#theme'] = 'notify_widget_block';
    $build['#notifications'] = $notifications;
    $build['#unread_count'] = $unreadCount;
    $build['#uid'] = $uid;
    $build['#base_url'] = $this->basePath;
    $build['#new_to_notifications'] = !($newToNotifications === 'no');

    // If the user had not seen the notifications block prior to this build
    // they now have, so we can set a flag in their userdata to indicate this.
    // We only set this flag if there are no notifications to show.
    if ($newToNotifications && !count($notifications)) {
      $this->userData->set('notify_widget', $uid, 'new_to_notifications', 'no');
    }

    // Attach JS library for pop-up, if configured to use module CSS.
    if ($this->configFactory->get('notify_widget.settings')->get('use_module_css') ?? 1) {
      $build['#attached']['library'][] = 'notify_widget/notifications_popup_css';
    }
    else {
      $build['#attached']['library'][] = 'notify_widget/notifications_popup';
    }

    // Cache the block by user ID.
    $build['#cache'] = [
      'tags' => ['notify_widget:' . $this->currentUser->id()],
    ];

    return $build;
  }

  /**
   * Convert a timestamp to time-ago format.
   *
   * @param int|string $ts
   *   The timestamp to convert.
   *
   * @return string
   *   Relative time.
   */
  private function relativeTime(int|string $ts): string {
    if (!ctype_digit($ts)) {
      $ts = strtotime($ts);
    }

    $diff = time() - $ts;
    if ($diff == 0) {
      return $this->t('now');
    }
    elseif ($diff > 0) {
      $day_diff = floor($diff / 86400);
      if ($day_diff == 0) {
        if ($diff < 60) {
          return $this->t('just now');
        }
        if ($diff < 120) {
          return $this->t('1 minute ago');
        }
        if ($diff < 3600) {
          return $this->t('@minutes minutes ago', ['@minutes' => floor($diff / 60)]);
        }
        if ($diff < 7200) {
          return $this->t('1 hour ago');
        }
        if ($diff < 86400) {
          return $this->t('@hours hours ago', ['@hours' => floor($diff / 3600)]);
        }
      }
      if ($day_diff == 1) {
        return $this->t('Yesterday');
      }
      if ($day_diff < 7) {
        return $this->t('@day days ago', ['@day' => $day_diff]);
      }
      if ($day_diff < 14) {
        return $this->t('1 week ago');
      }
      if ($day_diff < 31) {
        return $this->t('@weeks weeks ago', ['@weeks' => ceil($day_diff / 7)]);
      }
      if ($day_diff < 60) {
        return $this->t('last month');
      }
      return date('F Y', (int) $ts);
    }
    else {
      $diff = abs($diff);
      $day_diff = floor($diff / 86400);
      if ($day_diff == 0) {
        if ($diff < 120) {
          return $this->t('in a minute');
        }
        if ($diff < 3600) {
          return $this->t('in @minute minutes', ['@minute' => floor($diff / 60)]);
        }
        if ($diff < 7200) {
          return $this->t('in an hour');
        }
        if ($diff < 86400) {
          return $this->t('in @hours hours', ['@hours' => floor($diff / 3600)]);
        }
      }
      if ($day_diff == 1) {
        return $this->t('Tomorrow');
      }
      if ($day_diff < 4) {
        return date('l', $ts);
      }
      if ($day_diff < 7 + (7 - date('w'))) {
        return $this->t('next week');
      }
      if (ceil($day_diff / 7) < 4) {
        return $this->t('in @weeks weeks', ['@weeks' => ceil($day_diff / 7)]);
      }
      if (date('n', $ts) == date('n') + 1) {
        return $this->t('next month');
      }
      return date('F Y', $ts);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['renderNotifyWidgetBlock'];
  }

}
