<?php

namespace Drupal\number_element\Element;

use Drupal\Core\Render\Element\Textfield;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines a 'number_element' form element.
 * 
 * Usage example:
 * 
 * @code
 * $form['number_element'] = [
 *   '#type' => 'number_element',
 *   '#min' => 1,
 *   '#max' => 2,
 *   '#precision' => 3,
 *   '#leading_zeros' => false,
 *   '#message' => 'Please enter a valid number.',
 * ];
 * @endcode
 * 
 * @FormElement("number_element")
 */
class NumberElement extends TextField
{

    /**
     * {@inheritdoc}
     */
    public function getInfo()
    {
        $info = parent::getInfo();
        $info['#element_validate'][] = [get_class($this), 'validateNumber'];
        return $info;
    }

    /**
     * {@inheritdoc}
     */
    public static function validateNumber(array &$element, FormStateInterface $form_state)
    {
        $key = $element['#webform_key'];
        $value = $form_state->getValue($element['#name']);
        // Message can also be set in the element.
        $message = $element['#message'] ?? t('Must be a positive number.');

        // Checks for valid numeric value, currently will not accept commas as decimal separator. 
        // (Possibly something to be changed later?)
        if (!is_numeric($value) && !empty($value)) {
            $form_state->setError($element, $message);
        }

        // If there are no leading zeros, a leading zero will be added to the value
        // after the user continues and validation is complete.
        if (!empty($value)) {
            if ($value[0] === '.') {
                $value = '0' . $value;
                $form_state->setValue($key, $value);
            }
        }

        // Call each function to validate the number for each property.
        self::validateMaxMin($element, $form_state, $value, $message);
        self::validateLeadingZeros($element, $form_state, $value, $message);
        self::validatePrecision($element, $form_state, $value, $message);
    }

    /**
     * {@inheritdoc}
     */
    public static function validateMaxMin(array &$element, FormStateInterface $form_state, $value, $message)
    {
        $max_value = $element['#max'] ?? null;
        $min_value = $element['#min'] ?? null;

        if ($max_value !== null && $value > $max_value) {
            $form_state->setError($element, $message);
        }

        if ($min_value !== null && $value < $min_value) {
            $form_state->setError($element, $message);
        }
    }

    /**
     * {@inheritdoc}
     */
    public static function validateLeadingZeros(array &$element, FormStateInterface $form_state, $value, $message)
    {
        // Leading zeros can be allowed or disallowed, they're allowed by default.
        $allow_leading_zeros = $element['#leading_zeros'] ?? TRUE;
        $string_value = trim((string) $value);

        if ($allow_leading_zeros) {
            $parts = explode('.', $string_value, 2);
            $integer_half = $parts[0];

            if ($integer_half === '0') {
                return;
            }

            if (strlen($integer_half) > 1 && $integer_half[0] === '0') {
                $form_state->setError($element, $message);
            }
        }

        if (strpos($string_value, '.') === 0) {
            return;
        }
        if (strlen($string_value) > 1 && $string_value[0] === '0') {
            $form_state->setError($element, $message);
        }
    }

    /**
     * {@inheritdoc}
     */
    public static function validatePrecision(array &$element, FormStateInterface $form_state, $value, $message)
    {
        // The maximum default precision is 3 decimal places.
        $precision = $element['#precision'] ?? 3;

        $decimal_halves = explode('.', $value);
        if (!isset($decimal_halves[1]) || strlen(rtrim($decimal_halves[1], '0')) <= $precision) {
            return;
        }
        $form_state->setError($element, $message);
    }
}
