<?php

declare(strict_types=1);

namespace Drupal\number_range_slider\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines the 'number_range_slider' field widget.
 *
 * @FieldWidget(
 *   id = "number_range_slider",
 *   label = @Translation("Number Slider"),
 *   field_types = {"integer", "decimal", "float"},
 * )
 */
final class NumberRangeSliderWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $setting = [
      'step' => 1,
      'displayValue' => FALSE,
      'isPercentage' => FALSE,
    ];
    return $setting + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    // @todo Surely there's a better way to dinamically get the minimum allowed
    // step value here, based on decimal scale.
    $scale = $this->fieldDefinition->getFieldStorageDefinition()->getSetting('scale');
    $step = isset($scale) ? "0." . str_repeat("0", $scale - 1) . "1" : 1;

    $element['step'] = [
      '#type' => 'number',
      '#required' => TRUE,
      '#title' => $this->t('Step'),
      '#step' => (float) $step,
      '#min' => $this->fieldDefinition->getSetting('min'),
      '#max' => $this->fieldDefinition->getSetting('max'),
      '#description' => $this->t('The value for what each step between the slider should be.'),
      '#default_value' => $this->getSetting('step'),
    ];

    $element['displayValue'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Display value"),
      '#description' => $this->t("If the value should be displayed as the user slides. 
            Adds a small Javascript snippet to add that feature, as it's not otherwise natively supported."),
      '#default_value' => $this->getSetting('displayValue') ?? FALSE,
    ];

    if ($this->fieldDefinition->getType() === 'decimal') {
      $element['isPercentage'] = [
        '#type' => 'checkbox',
        '#title' => $this->t("Percentage"),
        '#description' => $this->t("If the value is a percentage. So that when we display its value, we multiply it by 100.
            This does not actually change the final submitted value, just the value we display for the slider."),
        '#states' => [
          'visible' => [
            ':input[name*="displayValue"]' => ['checked' => TRUE],
          ],
        ],
        '#default_value' => $this->getSetting('isPercentage') ?? FALSE,
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [$this->t('Step: @step', ['@step' => $this->getSetting('step')])];

    $displayValue = (bool) $this->getSetting('displayValue') ?? FALSE;
    $summary[] = $this->t('Display value: @displayValue', ['@displayValue' => $this->convertCheckboxValueToText($displayValue)]);

    if ($this->fieldDefinition->getType() === 'decimal' && $displayValue === TRUE) {
      $summary[] = $this->t('Percentage: @isPercentage', ['@isPercentage' => $this->convertCheckboxValueToText($this->getSetting('isPercentage'))]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element['value'] = $element + [
      '#type' => 'range',
      '#min' => $this->fieldDefinition->getSetting('min'),
      '#max' => $this->fieldDefinition->getSetting('max'),
      '#step' => $this->getSetting('step'),
      '#default_value' => $items[$delta]->value ?? NULL,
      '#attributes' => ['class' => ['slider-input']],
    ];

    if ($this->getSetting('displayValue') != TRUE) {
      return $element;
    }

    $element['#attached']['library'][] = 'number_range_slider/range_slider';
    $element['#attached']['drupalSettings']['number_range_slider'] = [];

    $prefix = $this->fieldDefinition->getSetting('prefix');
    if (!is_null($prefix)) {
      $element['#attached']['drupalSettings']['number_range_slider']['prefix'] = $prefix;
    }

    $suffix = $this->fieldDefinition->getSetting('suffix');
    if (!is_null($suffix)) {
      $element['#attached']['drupalSettings']['number_range_slider']['suffix'] = $suffix;
    }

    $isPercentage = $this->getSetting('isPercentage');
    if ($isPercentage == TRUE) {
      $element['#attached']['drupalSettings']['number_range_slider']['isPercentage'] = $isPercentage;
    }

    return $element;
  }

  /**
   * Convert checkbox values to "Yes" or "No" text values.
   *
   * @param string|null $value
   *   The checkbox value.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The textual representation of the boolean value.
   */
  private function convertCheckboxValueToText(mixed $value): TranslatableMarkup {
    if ((bool) $value === TRUE) {
      return $this->t("Yes");
    }

    return $this->t("No");
  }

}
