<?php

declare(strict_types=1);

namespace Drupal\nva\Plugin\Block;

use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Stinis\Nva\APIClient;

/**
 * Provides a list publications block.
 */
#[Block(
  id: 'nva_list_publications_ajax',
  admin_label: new TranslatableMarkup('List publications Ajax'),
  category: new TranslatableMarkup('NVA'),
)]
final class ListPublicationsAjaxBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The NVA API client.
   *
   * @var \Stinis\Nva\APIClient
   */
  private APIClient $apiClient;

  /**
   * The pager manager.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  private PagerManagerInterface $pagerManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  private ModuleHandlerInterface $moduleHandler;

  public function __construct(array $configuration, $plugin_id, array $plugin_definition, APIClient $apiClient, PagerManagerInterface $pagerManager, ModuleHandlerInterface $moduleHandler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiClient = $apiClient;
    $this->pagerManager = $pagerManager;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new ListPublicationsAjaxBlock(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('nva.api_client'),
      $container->get('pager.manager'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form['authorName'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Author Name'),
      '#default_value' => $this->configuration['authorName'] ?? '',
      '#required' => TRUE,
    ];
    $form['results_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of results per page'),
      '#default_value' => $this->configuration['results_per_page'] ?? 10,
      '#description' => $this->t('If left empty, all publications will be shown.'),
    ];
    // Add sort by radio buttons.
    $form['sort_by'] = [
      '#type' => 'radios',
      '#title' => $this->t('Sort by'),
      '#options' => [
        'title' => $this->t('Title'),
        'published_date' => $this->t('Published date'),
      ],
      '#default_value' => $this->configuration['sort_by'] ?? 'published_date',
    ];
    $form['categories'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Categories'),
      '#options' => [
        'AcademicArticle' => $this->t('Academic Article'),
        'AcademicChapter' => $this->t('Academic Chapter'),
        'AcademicLiteratureReview' => $this->t('Academic Literature Review'),
        'AcademicMonograph' => $this->t('Academic Monograph'),
        'Architecture' => $this->t('Architecture'),
        'ArtisticDesign' => $this->t('Artistic Design'),
        'BookAbstracts' => $this->t('Book Abstracts'),
        'BookAnthology' => $this->t('Book Anthology'),
        'BookMonograph' => $this->t('Book Monograph'),
        'CaseReport' => $this->t('Case Report'),
        'ChapterArticle' => $this->t('Chapter Article'),
        'ChapterConferenceAbstract' => $this->t('Chapter Conference Abstract'),
        'ChapterInReport' => $this->t('Chapter In Report'),
        'ConferenceAbstract' => $this->t('Conference Abstract'),
        'ConferenceLecture' => $this->t('Conference Lecture'),
        'ConferencePoster' => $this->t('Conference Poster'),
        'ConferenceReport' => $this->t('Conference Report'),
        'DataManagementPlan' => $this->t('Data Management Plan'),
        'DataSet' => $this->t('Data Set'),
        'DegreeBachelor' => $this->t('Degree Bachelor'),
        'DegreeLicentiate' => $this->t('Degree Licentiate'),
        'DegreeMaster' => $this->t('Degree Master'),
        'DegreePhd' => $this->t('Degree PhD'),
        'Encyclopedia' => $this->t('Encyclopedia'),
        'EncyclopediaChapter' => $this->t('Encyclopedia Chapter'),
        'ExhibitionCatalog' => $this->t('Exhibition Catalog'),
        'ExhibitionCatalogChapter' => $this->t('Exhibition Catalog Chapter'),
        'ExhibitionProduction' => $this->t('Exhibition Production'),
        'FeatureArticle' => $this->t('Feature Article'),
        'Introduction' => $this->t('Introduction'),
        'JournalArticle' => $this->t('Journal Article'),
        'JournalCorrigendum' => $this->t('Journal Corrigendum'),
        'JournalInterview' => $this->t('Journal Interview'),
        'JournalIssue' => $this->t('Journal Issue'),
        'JournalLeader' => $this->t('Journal Leader'),
        'JournalLetter' => $this->t('Journal Letter'),
        'JournalReview' => $this->t('Journal Review'),
        'Lecture' => $this->t('Lecture'),
        'LiteraryArts' => $this->t('Literary Arts'),
        'MediaBlogPost' => $this->t('Media Blog Post'),
        'MediaFeatureArticle' => $this->t('Media Feature Article'),
        'MediaInterview' => $this->t('Media Interview'),
        'MediaParticipationInRadioOrTv' => $this->t('Media Participation In Radio Or TV'),
        'MediaReaderOpinion' => $this->t('Media Reader Opinion'),
        'MovingPicture' => $this->t('Moving Picture'),
        'MusicPerformance' => $this->t('Music Performance'),
        'NonFictionChapter' => $this->t('Non Fiction Chapter'),
        'NonFictionMonograph' => $this->t('Non Fiction Monograph'),
        'OtherPresentation' => $this->t('Other Presentation'),
        'OtherStudentWork' => $this->t('Other Student Work'),
        'PerformingArts' => $this->t('Performing Arts'),
        'PopularScienceArticle' => $this->t('Popular Science Article'),
        'PopularScienceChapter' => $this->t('Popular Science Chapter'),
        'PopularScienceMonograph' => $this->t('Popular Science Monograph'),
        'ProfessionalArticle' => $this->t('Professional Article'),
        'ReportBasic' => $this->t('Report Basic'),
        'ReportBookOfAbstract' => $this->t('Report Book Of Abstract'),
        'ReportPolicy' => $this->t('Report Policy'),
        'ReportResearch' => $this->t('Report Research'),
        'ReportWorkingPaper' => $this->t('Report Working Paper'),
        'StudyProtocol' => $this->t('Study Protocol'),
        'Textbook' => $this->t('Textbook'),
        'TextbookChapter' => $this->t('Textbook Chapter'),
        'VisualArts' => $this->t('Visual Arts'),
      ],
      '#default_value' => $this->configuration['categories'] ?? [],
      '#description' => $this->t('Select the categories you want to include in the list. If none are selected, all categories will be included.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['authorName'] = $form_state->getValue('authorName');
    $this->configuration['results_per_page'] = $form_state->getValue('results_per_page');
    $this->configuration['sort_by'] = $form_state->getValue('sort_by');
    $this->configuration['categories'] = array_filter($form_state->getValue('categories'));
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $imagePath = $this->moduleHandler->getModule('nva')->getPath() . '/assets/images/spinner.gif';

    $build['content'] = [
      '#markup' => '<div id="nva_publications_ajax"><img src="/' . $imagePath . '" width="30" height="30"></div>',
      '#pager' => [
        '#type' => 'pager',
      ],
      '#attached' => [
        'library' => [
          'nva/nva-ajax',
        ],
        'drupalSettings' => [
          'nva' => [
            'authorName' => $this->configuration['authorName'],
            'resultsPerPage' => $this->configuration['results_per_page'],
            'sortBy' => $this->configuration['sort_by'] ?? '',
            'categories' => $this->configuration['categories'] ?? '',
          ],
        ],
      ],
    ];
    return $build;
  }

}
