<?php

declare(strict_types=1);

namespace Drupal\nva\Plugin\Block;

use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\nva\Service\ApiClientConfigurator;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Stinis\Nva\APIClient;

/**
 * Provides a list of projects by user.
 */
#[Block(
  id: 'nva_projects_by_user',
  admin_label: new TranslatableMarkup('List projects by user'),
  category: new TranslatableMarkup('NVA'),
)]
final class ProjectsByUserBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The NVA API client.
   *
   * @var \Stinis\Nva\APIClient
   */
  private APIClient $apiClient;

  /**
   * The API client configurator.
   *
   * @var \Drupal\nva\Service\ApiClientConfigurator
   */
  private ApiClientConfigurator $apiClientConfigurator;

  public function __construct(array $configuration, $plugin_id, array $plugin_definition, ApiClientConfigurator $apiClientConfigurator) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $apiClient = $apiClientConfigurator->getConfiguredApiClient();
    $this->apiClient = $apiClient;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new ProjectsByUserBlock(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('nva.api_client_configurator'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form['authorName'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Author Name'),
      '#default_value' => $this->configuration['authorName'] ?? '',
      '#required' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['authorName'] = $form_state->getValue('authorName');
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $authorName = $variables['authorName'] ?? $this->configuration['authorName'] ?? '';
    // Gets the person ID from the API.
    $person = $this->apiClient->getCristinPersonByQueryParameters(['name' => $authorName]);
    if (isset($person['hits'][0])) {
      // If the person is found, get the person ID.
      $cristinID = $person['hits'][0]['identifiers'][0]['value'];
      $queryParameters = [
        'participant' => $cristinID,
        'results' => 10000,
      ];
      // Get projects from the API.
      $projects = $this->apiClient->getCristinProjectByQueryParameters($queryParameters);
      $projects = $projects['hits'];
    }

    $build['content'] = [
      '#theme' => 'nva_projects_by_user',
      '#projects' => $projects ?? [],
    ];
    return $build;
  }

}
