<?php

namespace Drupal\o365\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\o365\O365ConnectorInterface;

/**
 * Defines the Microsoft 365 connector entity.
 *
 * @ConfigEntityType(
 *   id = "o365_connector",
 *   label = @Translation("Microsoft 365 connector"),
 *   handlers = {
 *     "list_builder" = "Drupal\o365\Controller\O365ConnectorListBuilder",
 *     "access" = "Drupal\o365\O365ConnectorAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\o365\Form\O365ConnectorForm",
 *       "edit" = "Drupal\o365\Form\O365ConnectorForm",
 *       "delete" = "Drupal\o365\Form\O365ConnectorDeleteForm"
 *     }
 *   },
 *   config_prefix = "o365_connector",
 *   admin_permission = "administer o365 connectors",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "status" = "status",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "status",
 *     "redirect_login",
 *     "auth_scopes",
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/system/o365-connectors/{o365_connector}",
 *     "delete-form" = "/admin/config/system/o365-connectors/{o365_connector}/delete"
 *   }
 * )
 */
class O365Connector extends ConfigEntityBase implements O365ConnectorInterface {

  /**
   * The O365Connector ID.
   */
  protected string $id;

  /**
   * The O365Connector label.
   */
  protected string $label;

  /**
   * The redirect URL string.
   */
  protected string $redirect_login;

  /**
   * The auth scopes.
   */
  protected string $auth_scopes;

  /**
   * {@inheritdoc}
   */
  public function getClientId(): string {
    return $this->getValuesFromConfig('client_id') ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getClientSecret(): string {
    return $this->getValuesFromConfig('client_secret') ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getTenantId(): string {
    return $this->getValuesFromConfig('tenant_id') ?? 'common';
  }

  /**
   * {@inheritdoc}
   */
  public function getRedirectLogin(): string {
    $redirectLogin = $this->getValuesFromConfig('redirect_login');
    if ($redirectLogin) {
      return $redirectLogin;
    }

    return $this->redirect_login ?? '';
  }

  /**
   * Get values from the config in settings.php.
   *
   * @param string $key
   *   The config key name.
   *
   * @return false|mixed
   *   The configured value or false.
   */
  private function getValuesFromConfig(string $key) {
    /** @var \Drupal\o365\HelperService $helperService */
    $helperService = \Drupal::service('o365.helpers');
    $config = $helperService->getApiConfig($this->id());

    return $config[$key] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthScopes(): string {
    return $this->auth_scopes ?? '';
  }

}
