<?php

namespace Drupal\o365_sharepoint_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\o365\AuthenticationServiceInterface;
use Drupal\o365\GraphService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\ConstraintViolationInterface;

/**
 * Defines the 'field_sharepoint_search_link' field widget.
 *
 * @FieldWidget(
 *   id = "sharepoint_search_link",
 *   label = @Translation("Microsoft 365 - SharePoint Search field"),
 *   field_types = {"sharepoint_search_link"},
 * )
 */
class SharePointSearchWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The authentication service.
   *
   * @var \Drupal\o365\AuthenticationServiceInterface
   */
  protected AuthenticationServiceInterface $authenticationService;

  /**
   * The graph service.
   *
   * @var \Drupal\o365\GraphService
   */
  protected GraphService $graphService;

  /**
   * Constructs a SharePointSearchWidget object.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   The third party settings.
   * @param \Drupal\o365\AuthenticationServiceInterface $authenticationService
   *   The authentication service.
   * @param \Drupal\o365\GraphService $graphService
   *   The graph service.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, AuthenticationServiceInterface $authenticationService, GraphService $graphService) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->authenticationService = $authenticationService;
    $this->graphService = $graphService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    // @phpstan-ignore-next-line
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('o365.authentication'),
      $container->get('o365.graph')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    if ($this->authenticationService->checkForOfficeLogin()) {
      $form['#attached']['library'][] = 'core/drupal.ajax';
      $form['#attached']['library'][] = 'o365_sharepoint_field/autocomplete';

      $value = $items->getValue();
      $sharepointSearch = $value[0]['sharepoint_file_name'] ?? '';
      $hitId = $value[0]['sharepoint_hit_id'] ?? '';
      $driveId = $value[0]['sharepoint_drive_id'] ?? '';
      $webUrl = $value[0]['sharepoint_web_url'] ?? '';

      $element['sharepoint_file_name'] = [
        '#type' => 'textfield',
        '#title' => $element['#title'],
        '#maxlength' => 255,
        '#default_value' => $sharepointSearch,
        '#autocomplete_route_name' => 'o365_sharepoint_field.autocomplete.search_sharepoint',
        '#attributes' => [
          'class' => ['o365-sharepoint-autocomplete'],
        ],
      ];

      $element['sharepoint_hit_id'] = [
        '#type' => 'hidden',
        '#title' => $this->t('HitId'),
        '#maxlength' => 255,
        '#default_value' => $hitId,
        '#attributes' => [
          'class' => ['o365-sharepoint-hit-id'],
        ],
      ];

      $element['sharepoint_drive_id'] = [
        '#type' => 'hidden',
        '#title' => $this->t('DriveId'),
        '#maxlength' => 255,
        '#default_value' => $driveId,
        '#attributes' => [
          'class' => ['o365-sharepoint-drive-id'],
        ],
      ];

      $element['sharepoint_web_url'] = [
        '#type' => 'hidden',
        '#title' => $this->t('Web url'),
        '#maxlength' => 6000,
        '#default_value' => $webUrl,
      ];
    }
    else {
      $element['sharepoint_message'] = [
        '#type' => 'markup',
        '#markup' => $this->t('You are not logged in with Office. This field is disabled because of that.'),
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function errorElement(array $element, ConstraintViolationInterface $violation, array $form, FormStateInterface $form_state) {
    return isset($violation->arrayPropertyPath[0]) ? $element[$violation->arrayPropertyPath[0]] : $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    if ($values[0]['sharepoint_drive_id'] !== '' && $values[0]['sharepoint_hit_id'] !== '') {
      // Filling the web_url field based on drive_id and hit_id.
      if ($this->authenticationService->checkForOfficeLogin()) {
        $sharepointImageData = $this->graphService->getGraphData('/drives/' . $values[0]['sharepoint_drive_id'] . '/items/' . $values[0]['sharepoint_hit_id']);
        $sharepointUrl = $sharepointImageData['webUrl'];
        $values[0]['sharepoint_web_url'] = $sharepointUrl;
      }
    }

    return $values;
  }

}
