<?php

namespace Drupal\o365\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Builds the form to create/edit Microsoft 365 connector entities.
 */
class O365ConnectorForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\o365\O365ConnectorInterface $o365_connector */
    $o365_connector = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#default_value' => $o365_connector->label(),
      '#maxlength' => 64,
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $o365_connector->id(),
      '#machine_name' => [
        'exists' => [
          $this,
          'exist',
        ],
      ],
      '#disabled' => !$o365_connector->isNew(),
    ];

    $form['redirect_login'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Redirect after login URL'),
      '#description' => $this->t('After login users will be redirected to this page. This needs to be a full url like https://www.example.com.'),
      '#maxlength' => 255,
      '#size' => 64,
      '#default_value' => $o365_connector->getRedirectLogin(),
      '#required' => TRUE,
    ];

    $added_scopes = [];
    $this->moduleHandler->invokeAll('o365_auth_scopes', [
      &$added_scopes,
      $o365_connector,
    ]);

    $form['auth_scopes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Authorization scopes'),
      '#description' => $this->t('A space separated list of authorization scopes. The offline_access scope (needed for login purposes) is automatically added. <br>Following scopes are automatically added: <b>@scopes</b>', ['@scopes' => implode(', ', $added_scopes)]),
      '#default_value' => $o365_connector->getAuthScopes(),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $o365_connector->status(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $status = $this->entity->save();

    if ($status === SAVED_NEW) {
      $this->messenger()->addMessage($this->t('The @label connector created.', [
        '@label' => $this->entity->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('The @label connector updated.', [
        '@label' => $this->entity->label(),
      ]));
    }

    $form_state->setRedirect('entity.o365_connector.collection');

    return $status;
  }

  /**
   * Helper function to check whether a Social o365 configuration entity exists.
   */
  public function exist($id): bool {
    $entity = $this->entityTypeManager->getStorage('o365_connector')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('id', $id)
      ->execute();

    return (bool) $entity;
  }

}
