<?php

declare(strict_types=1);

namespace Drupal\o365\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\Role;

/**
 * Map groups in Microsoft 365 to Drupal roles.
 */
final class RoleSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'o365_role_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['o365.role_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('o365.role_settings');

    $roles = Role::loadMultiple();
    $defaultRoleOptions = [];
    /** @var \Drupal\user\Entity\Role $role */
    foreach ($roles as $role) {
      $defaultRoleOptions[$role->id()] = $role->label();
    }

    $form['default_role'] = [
      '#type' => 'select',
      '#title' => $this->t('Default role'),
      '#description' => $this->t('Select the default role when logging in.'),
      '#default_value' => $config->get('default_role'),
      '#options' => $defaultRoleOptions,
      '#empty_option' => $this->t('None'),
    ];

    $form['roles_map'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Role mapping'),
      '#description' => $this->t('Put a mapping on each line, in format [Group in O365]|[Drupal role name]|[Another Drupal role name].<br/>For example: <i>O365_TEACHER|teacher</i>. For the group name in Microsoft 365 you can use the id, displayName or onPremisesSamAccountName.'),
      '#default_value' => $config->get('roles_map'),
    ];

    $form['safe_roles'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Safe roles'),
      '#description' => $this->t('These are the roles the module won\'t remove from the logged in user. The "anonymous" and "authenticated" roles are already excluded.'),
      '#default_value' => $config->get('safe_roles'),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('o365.role_settings')
      ->set('roles_map', trim($form_state->getValue('roles_map')))
      ->set('safe_roles', $form_state->getValue('safe_roles'))
      ->set('default_role', $form_state->getValue('default_role'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
