<?php

namespace Drupal\o365;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Site\Settings;

/**
 * Service with some helpful methods that we use in different locations.
 */
class HelperService {

  /**
   * Drupal\Core\Datetime\DateFormatterInterface definition.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new HelperService object.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The DateFormatterInterface definition.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(DateFormatterInterface $date_formatter, ModuleHandlerInterface $moduleHandler) {
    $this->dateFormatter = $date_formatter;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * Create a ISO8601 timestamp that Microsoft Graph API can use.
   *
   * @param int $timestamp
   *   The unix timestamp we want to use to create the date.
   *
   * @return string
   *   The ISO8601 formatted date.
   */
  public function createIsoDate(int $timestamp): string {
    return $this->dateFormatter->format($timestamp, 'custom', 'Y-m-d\TH:i:s.000') . 'Z';
  }

  /**
   * Format a ISO8601 date into something more readable.
   *
   * @param string $date
   *   The ISO8601 date.
   * @param string $timezone
   *   The timezone.
   * @param string $format
   *   The format we want to convert to.
   *
   * @return string
   *   The formatted date.
   *
   * @throws \Exception
   */
  public function formatDate(string $date, string $timezone = 'UTC', string $format = 'd-m-Y H:i'): string {
    $dateTimezone = new \DateTimeZone($timezone);
    $dateTime = new \DateTime($date, $dateTimezone);
    $timestamp = (int) $dateTime->format('U');

    return $this->dateFormatter->format($timestamp, 'custom', $format);
  }

  /**
   * Retrieve the API settings from the config.
   *
   * @return false|array
   *   The array with the settings, or FALSE.
   */
  public function getApiConfig(?string $config_id = 'default') {
    $settings = Settings::get('o365');
    return $settings[$config_id] ?? FALSE;
  }

  /**
   * Get the auth scopes string.
   *
   * @param \Drupal\o365\O365ConnectorInterface|null $o365_connector
   *   The o365 connector.
   * @param bool $asArray
   *   If the returned value needs to be a array, defaults to FALSE.
   *
   * @return string|array
   *   The space seperated list of auth scopes or a raw array of scopes.
   */
  public function getAuthScopes(?O365ConnectorInterface $o365_connector, bool $asArray = FALSE) {
    // Get the scopes from the entity.
    if (!$o365_connector) {
      $scopes = [];
    }
    else {
      $scopes = explode(' ', $o365_connector->getAuthScopes());
    }

    // Get the scopes from the hook implementations.
    $this->moduleHandler->invokeAll('o365_auth_scopes', [
      &$scopes,
      $o365_connector,
    ]);

    // Remove duplicate and empty values.
    $scopes = array_unique($scopes);
    $scopes = array_filter($scopes);

    // We need to check for the offline_access value.
    if (!in_array('offline_access', $scopes, TRUE)) {
      $scopes[] = 'offline_access';
    }

    // Return as array.
    if ($asArray) {
      return $scopes;
    }

    // Return a exploded string.
    return implode(' ', $scopes);
  }

  /**
   * Generate a unix timestamp from a date.
   *
   * @param string $date
   *   The date.
   * @param string $timezone
   *   The timezone.
   *
   * @return string
   *   The unix timestamp.
   */
  public function getTsFromDate(string $date, string $timezone = 'UTC'): string {
    $dateTimezone = new \DateTimeZone($timezone);
    $dateTime = new \DateTime($date, $dateTimezone);

    return $dateTime->format('U');
  }

  /**
   * Checks if a string starts with a given substring.
   *
   * Already present in PHP8, but we also need to work on PHP < 8
   *
   * @param string $haystack
   *   The string to search in.
   * @param string $needle
   *   The substring to search for in the haystack.
   *
   * @return bool
   *   Returns true if haystack begins with needle, false otherwise.
   */
  public function strStartsWith(string $haystack, string $needle): bool {
    return strpos($haystack, $needle) === 0;
  }

}
