<?php

namespace Drupal\o365\Plugin\Oauth2Client;

use Drupal\o365\HelperService;
use Drupal\oauth2_client\Plugin\Oauth2Client\Oauth2ClientPluginBase;
use League\OAuth2\Client\Token\AccessTokenInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * OAuth2 Client to authenticate with Microsoft 365.
 *
 * @Oauth2Client(
 *   id = "o365",
 *   name = @Translation("Microsoft 365"),
 *   grant_type = "client_credentials",
 *   client_id = "",
 *   client_secret = "",
 *   authorization_uri =
 *   "https://login.microsoftonline.com/common/oauth2/v2.0/authorize",
 *   token_uri = "https://login.microsoftonline.com/common/oauth2/v2.0/token",
 *   resource_owner_uri = "",
 *   credential_provider = "o365_sso",
 *   storage_key = "",
 * )
 */
class O365OAuth2Client extends Oauth2ClientPluginBase {

  /**
   * The o365 helper service.
   *
   * @var \Drupal\o365\HelperService
   */
  protected HelperService $helperService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->helperService = $container->get('o365.helpers');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientId(): string {
    $client_id = $this->helperService->getApiConfig()['client_id'] ?? NULL;
    if (empty($client_id)) {
      throw new \LogicException('The o365 OAuth2 Client id cannot be empty.');
    }
    return $client_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientSecret(): string {
    $client_secret = $this->helperService->getApiConfig()['client_secret'] ?? NULL;
    if (empty($client_secret)) {
      throw new \LogicException('The o365 OAuth2 Client secret cannot be empty.');
    }
    return $client_secret;
  }

  /**
   * {@inheritdoc}
   */
  public function storeAccessToken(AccessTokenInterface $accessToken): void {
    $this->state->set(
      'oauth2_client_access_token-' . $this->getId(),
      $accessToken
    );
  }

  /**
   * {@inheritdoc}
   */
  public function retrieveAccessToken(): AccessTokenInterface|NULL {
    return $this->state->get('oauth2_client_access_token-' . $this->getId());
  }

  /**
   * {@inheritdoc}
   */
  public function clearAccessToken(): void {
    $this->state->delete('oauth2_client_access_token-' . $this->getId());
  }

  /**
   * {@inheritdoc}
   */
  public function getCredentialProvider(): ?string {
    return 'o365_sso';
  }

}
