<?php

declare(strict_types=1);

namespace Drupal\o365\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\o365\GraphService;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\GuzzleException;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use Microsoft\Graph\Exception\GraphException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Microsoft 365 Connector form.
 */
final class DebuggerForm extends FormBase {

  /**
   * Constructs a new DebuggerForm object.
   *
   * @param \Drupal\o365\GraphService $graphService
   *   The o365 Graph service.
   */
  public function __construct(
    private readonly GraphService $graphService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('o365.graph')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'o365_debugger';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $url = Url::fromUri('https://developer.microsoft.com/en-us/graph/graph-explorer', ['attributes' => ['target' => '_blank']]);
    $link = Link::fromTextAndUrl('Graph Explorer', $url);
    $form['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GraphAPI endpoint'),
      '#required' => TRUE,
      '#description' => $this->t('The endpoint for Graph API requests. You can only debug the GET requests. For instance <i>/me</i> for information about your user. You can also check the @link for more information.', ['@link' => $link->toString()]),
      '#default_value' => '/me',
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Send'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $endpoint = $form_state->getValue('endpoint');

    $data = $this->graphService->getGraphData($endpoint);

    if ($data === FALSE || $data === NULL) {
      $this->messenger()->addWarning($this->t('No data returned from endpoint: @endpoint', ['@endpoint' => $endpoint]));
      return;
    }

    if (empty($data)) {
      $this->messenger()->addWarning($this->t('Empty response from endpoint: @endpoint', ['@endpoint' => $endpoint]));
      return;
    }

    // Format the output for better readability.
    $formatted_output = $this->formatGraphOutput($data);
    $this->messenger()->addStatus($this->t('Response from @endpoint:<br><pre>@data</pre>', [
      '@endpoint' => $endpoint,
      '@data' => $formatted_output,
    ]));
  }

  /**
   * Formats Graph API output for better readability.
   *
   * @param mixed $data
   *   The data to format.
   *
   * @return string
   *   The formatted output.
   */
  private function formatGraphOutput(mixed $data): string {
    if (is_array($data) || is_object($data)) {
      return json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }

    if (is_bool($data)) {
      return $data ? 'true' : 'false';
    }

    if (is_null($data)) {
      return 'null';
    }

    return (string) $data;
  }

}
