<?php

namespace Drupal\o365;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Some constants we use in the code.
 */
class ConstantsService {

  /**
   * The config factory interface.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The modules API config.
   */
  protected ImmutableConfig $apiConfig;

  /**
   * The request object.
   */
  protected Request $request;

  /**
   * The url where Microsoft will redirect us too.
   */
  private string $redirectUrl = '/o365/callback';

  /**
   * The authorize endpoint root.
   */
  private string $authorizeRoot = 'https://login.microsoftonline.com/';

  /**
   * The authorize endpoint path.
   */
  private string $authorizePath = '/oauth2/v2.0/authorize';

  /**
   * The token endpoint root.
   */
  private string $tokenRoot = 'https://login.microsoftonline.com/';

  /**
   * The token endpoint path.
   */
  private string $tokenPath = '/oauth2/v2.0/token';

  /**
   * The name of the temp store.
   */
  private string $userTempStoreName = 'o365.tempstore';

  /**
   * The name of the data saved in the temp store.
   */
  private string $userTempStoreDataName = 'o365AuthData';

  /**
   * Constructs a new ConstantsService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory interface.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack object.
   */
  public function __construct(ConfigFactoryInterface $configFactory, RequestStack $requestStack) {
    $this->configFactory = $configFactory;
    $this->apiConfig = $this->configFactory->get('o365.api_settings');
    $this->request = $requestStack->getCurrentRequest();
  }

  /**
   * Get the redirect URL.
   *
   * @return string
   *   The redirect url.
   */
  public function getRedirectUrl(): string {
    return 'https://' . $this->request->getHost() . $this->redirectUrl;
  }

  /**
   * Get the authorize url.
   *
   * @param \Drupal\o365\O365ConnectorInterface $o365_connector
   *   The o365 connector.
   *
   * @return string
   *   The authorize url.
   */
  public function getAuthorizeUrl(O365ConnectorInterface $o365_connector): string {
    $tenant = $o365_connector->getTenantId();
    return $this->authorizeRoot . $tenant . $this->authorizePath;
  }

  /**
   * Get the token url.
   *
   * @param \Drupal\o365\O365ConnectorInterface $o365_connector
   *   The o365 connector.
   *
   * @return string
   *   The token url.
   */
  public function getTokenUrl(O365ConnectorInterface $o365_connector): string {
    $tenant = $o365_connector->getTenantId();
    return $this->tokenRoot . $tenant . $this->tokenPath;
  }

  /**
   * Get the user temp store name.
   *
   * @return string
   *   The user temp store name.
   */
  public function getUserTempStoreName(): string {
    return $this->userTempStoreName;
  }

  /**
   * Get the user temp store data name.
   *
   * @return string
   *   The user temp store data name.
   */
  public function getUserTempStoreDataName(): string {
    return $this->userTempStoreDataName;
  }

}
