<?php

namespace Drupal\oauth\Authentication\Provider;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Authentication\AuthenticationProviderInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;
use Drupal\user\UserStorageInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;

/**
 * Oauth authentication provider.
 */
class OAuthDrupalProvider implements AuthenticationProviderInterface {

  /**
   * The database service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  private Connection $connection;

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  private UserDataInterface $userData;

  /**
   * The logger service for OAuth.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private LoggerInterface $logger;

  /**
   * An authenticated user object.
   *
   * @var \Drupal\user\UserInterface
   */
  private UserInterface $user;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  private TimeInterface $time;

  /**
   * The user entity storage handler.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  private UserStorageInterface $userStorage;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database service.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service for OAuth.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(Connection $connection, UserDataInterface $userData, LoggerInterface $logger, TimeInterface $time, EntityTypeManagerInterface $entityTypeManager) {
    $this->connection = $connection;
    $this->userData = $userData;
    $this->logger = $logger;
    $this->time = $time;
    $this->userStorage = $entityTypeManager->getStorage('user');
  }

  /**
   * {@inheritdoc}
   */
  public function applies(Request $request) {
    // Only check requests with the 'authorization' header starting with OAuth.
    return preg_match('/^OAuth/', $request->headers->get('authorization', ''));
  }

  /**
   * {@inheritdoc}
   */
  public function authenticate(Request $request) {
    try {
      // Initialize and configure the OauthProvider to handle the request.
      $provider = new \OAuthProvider();
      $provider->consumerHandler([$this, 'lookupConsumer']);
      $provider->timestampNonceHandler([$this, 'timestampNonceChecker']);
      $provider->tokenHandler([$this, 'tokenHandler']);
      $provider->is2LeggedEndpoint(TRUE);

      // Now check the request validity.
      $provider->checkOAuthRequest();
    }
    catch (\OAuthException $e) {
      // The OAuth extension throws an alert when there is something wrong
      // with the request (ie. the consumer key is invalid).
      $this->logger->warning($e->getMessage());
      return NULL;
    }

    // Check if we found a user.
    if (!empty($this->user)) {
      return $this->user;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function cleanup(Request $request) {}

  /**
   * {@inheritdoc}
   */
  public function handleException(ExceptionEvent $event) {
    return FALSE;
  }

  /**
   * Finds a user associated with the OAuth crendentials given in the request.
   *
   * For the moment it handles two legged authentication for a pair of
   * dummy key and secret, 'a' and 'b' respectively.
   *
   * @param \OAuthProvider $provider
   *   An instance of OauthProvider with the authorization request headers.
   *
   * @return int
   *   - OAUTH_OK if the authentication was successful.
   *   - OAUTH_CONSUMER_KEY_UNKNOWN if not.
   *
   * @see http://www.php.net/manual/en/class.oauthprovider.php
   */
  public function lookupConsumer(\OAuthProvider $provider) {
    $userData = $this->userData->get('oauth', NULL, $provider->consumer_key);
    if (!empty($userData)) {
      $provider->consumer_secret = $userData[key($userData)]['consumer_secret'];
      $this->user = $this->userStorage->load(key($userData));
      return OAUTH_OK;
    }
    else {
      return OAUTH_CONSUMER_KEY_UNKNOWN;
    }
  }

  /**
   * Token handler callback.
   *
   * @todo this will be used in token authorization.
   */
  public function tokenHandler($provider) {
    return OAUTH_OK;
  }

  /**
   * Nonce handler.
   */
  public function timestampNonceChecker($provider) {
    $query = $this->connection->select('oauth_nonce');
    $query->condition('nonce', $provider->nonce);
    $count = $query->countQuery()->execute()->fetchField();

    // If the nonce was found, then a request is being replayed.
    if ($count > 0) {
      return OAUTH_BAD_NONCE;
    }

    // Make note of the nonce so that it cannot be replayed later.
    $this->connection->insert('oauth_nonce')
      ->fields(['nonce', 'timestamp'], [$provider->nonce, $this->time->getRequestTime()])
      ->execute();

    return OAUTH_OK;
  }

}
