<?php

namespace Drupal\oauth\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Password\PasswordGeneratorInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to add OAuth consumers.
 */
class OAuthAddConsumerForm extends FormBase {

  const NAME = 'oauth_add_consumer_form';

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private AccountInterface $account;

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  private UserDataInterface $userData;

  /**
   * The password generator service.
   *
   * @var \Drupal\Core\Password\PasswordGeneratorInterface
   */
  private PasswordGeneratorInterface $passwordGenerator;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('user.data'),
      $container->get('password_generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return static::NAME;
  }

  /**
   * Constructs an OAuthAddConsumerForm.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user service.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Drupal\Core\Password\PasswordGeneratorInterface $passwordGenerator
   *   The password generator service.
   */
  public function __construct(AccountProxyInterface $account, UserDataInterface $userData, PasswordGeneratorInterface $passwordGenerator) {
    $this->account = $account;
    $this->userData = $userData;
    $this->passwordGenerator = $passwordGenerator;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, AccountInterface $user = NULL) {
    $form['save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add'),
    ];
    $form['uid'] = [
      '#type' => 'hidden',
      '#value' => $user ? $user->id() : $this->account->id(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $consumer_key = $this->passwordGenerator->generate(32);
    $consumer_secret = $this->passwordGenerator->generate(32);
    $key_hash = sha1($consumer_key);
    $uid = $form_state->getValue('uid');
    $consumer = [
      'consumer_secret' => $consumer_secret,
      'key_hash' => $key_hash,
    ];
    $this->userData->set('oauth', $uid, $consumer_key, $consumer);
    $this->messenger()->addStatus($this->t('Added a new consumer.'));
    Cache::invalidateTags(['oauth:' . $uid]);
    $form_state->setRedirect('oauth.user_consumer', ['user' => $uid]);
  }

}
