<?php

namespace Drupal\oauth\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\LinkGeneratorInterface;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller routines for oauth routes.
 */
class OAuthController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The URL generator service.
   *
   * @var \Drupal\Core\Utility\LinkGeneratorInterface
   */
  private LinkGeneratorInterface $linkGenerator;

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  private UserDataInterface $userData;

  /**
   * Constructs an OauthController object.
   *
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Drupal\Core\Utility\LinkGeneratorInterface $linkGenerator
   *   The link generator service.
   */
  public function __construct(UserDataInterface $userData, LinkGeneratorInterface $linkGenerator) {
    $this->userData = $userData;
    $this->linkGenerator = $linkGenerator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('user.data'),
      $container->get('link_generator')
    );
  }

  /**
   * Returns the list of consumers for a user.
   *
   * @param \Drupal\user\UserInterface $user
   *   A user account object.
   *
   * @return array
   *   The render array.
   */
  public function consumers(UserInterface $user): array {
    $list = [];

    $list['#cache']['tags'] = [
      'oauth:' => $user->id(),
    ];

    $list['heading']['#markup'] = $this->linkGenerator->generate($this->t('Add consumer'), Url::fromRoute('oauth.user_consumer_add', ['user' => $user->id()]));

    // Get the list of consumers.
    $result = $this->userData->get('oauth', $user->id());

    // Define table headers.
    $list['table'] = [
      '#theme' => 'table',
      '#header' => [
        'consumer_key' => [
          'data' => $this->t('Consumer key'),
        ],
        'consumer_secret' => [
          'data' => $this->t('Consumer secret'),
        ],
        'operations' => [
          'data' => $this->t('Operations'),
        ],
      ],
      '#rows' => [],
    ];

    // Add existing consumers to the table.
    foreach ($result as $key => $consumer) {
      $list['table']['#rows'][] = [
        'data' => [
          'consumer_key' => $key,
          'consumer_secret' => $consumer['consumer_secret'],
          'operations' => [
            'data' => [
              '#type' => 'operations',
              '#links' => [
                'delete' => [
                  'title' => $this->t('Delete'),
                  'url' => Url::fromRoute('oauth.user_consumer_delete', ['user' => $user->id(), 'key' => $key]),
                ],
              ],
            ],
          ],
        ],
      ];
    }

    $list['table']['#empty'] = $this->t('There are no OAuth consumers.');

    return $list;
  }

}
