<?php

namespace Drupal\oauth\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an oauth_consumer deletion confirmation form.
 */
class OAuthDeleteConsumerForm extends ConfirmFormBase implements ContainerInjectionInterface {

  const NAME = 'oauth_delete_consumer_form';

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private AccountInterface $account;

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  private UserDataInterface $userData;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('user.data')
    );
  }

  /**
   * Constructs an OAuthDeleteConsumerForm.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user service.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   */
  public function __construct(AccountProxyInterface $account, UserDataInterface $userData) {
    $this->account = $account;
    $this->userData = $userData;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormID() {
    return static::NAME;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete this OAuth consumer?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('oauth.user_consumer', ['user' => $this->account->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName() {
    return static::NAME;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?AccountInterface $user = NULL, $key = NULL) {
    $form['key'] = [
      '#type' => 'hidden',
      '#value' => $key,
    ];

    $form['uid'] = [
      '#type' => 'hidden',
      '#value' => $user->id(),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $key = $values['key'];
    $uid = $values['uid'];
    $this->userData->delete('oauth', $uid, $key);
    $this->messenger()->addStatus($this->t('OAuth consumer deleted.'));
    Cache::invalidateTags(['oauth:' . $uid]);
    $form_state->setRedirect('oauth.user_consumer', ['user' => $form_state->getValue('uid')]);
  }

}
