<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Entity;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface;
use Drupal\user\EntityOwnerInterface;
use Drupal\user\UserInterface;

/**
 * Provides an interface defining an OAuth Client Request entity type.
 */
interface OauthClientRequestInterface extends ContentEntityInterface, EntityChangedInterface, EntityOwnerInterface {

  /**
   * Sets the OAuth Client Request label.
   *
   * @param string $label
   *   The OAuth Client Request label.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setLabel(string $label): OauthClientRequestInterface;

  /**
   * Gets the associated client (consumer).
   *
   * @return \Drupal\consumers\Entity\ConsumerInterface|null
   *   The consumer entity or NULL if not set.
   */
  public function getClient(): ?ConsumerInterface;

  /**
   * Sets the associated client (consumer).
   *
   * @param \Drupal\consumers\Entity\ConsumerInterface|null $client
   *   The consumer entity or NULL.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setClient(?ConsumerInterface $client): OauthClientRequestInterface;

  /**
   * Gets the user who will own/use the client.
   *
   * @return \Drupal\user\UserInterface
   *   The user entity.
   */
  public function getUser(): UserInterface;

  /**
   * Sets the user who will own/use the client.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user entity.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setUser(UserInterface $user): OauthClientRequestInterface;

  /**
   * Gets the scope.
   *
   * @return \Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface|null
   *   The scope entity or NULL if not set.
   */
  public function getScope(): ?Oauth2ScopeEntityInterface;

  /**
   * Sets the scope.
   *
   * @param \Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface|null $scope
   *   The scope entity or NULL.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setScope(?Oauth2ScopeEntityInterface $scope): OauthClientRequestInterface;

  /**
   * Gets the request status.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestStatus
   *   The status enum.
   */
  public function getStatus(): OauthClientRequestStatus;

  /**
   * Sets the request status.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestStatus $status
   *   The status enum.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setStatus(OauthClientRequestStatus $status): OauthClientRequestInterface;

  /**
   * Checks if the request has a specific status.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestStatus $status
   *   The status to check.
   *
   * @return bool
   *   TRUE if the request has the specified status, FALSE otherwise.
   */
  public function isStatus(OauthClientRequestStatus $status): bool;

  /**
   * Gets the request reason.
   *
   * @return string|null
   *   The request reason or NULL if not set.
   */
  public function getRequestReason(): ?string;

  /**
   * Sets the request reason.
   *
   * @param string|null $reason
   *   The request reason or NULL.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setRequestReason(?string $reason): OauthClientRequestInterface;

  /**
   * Gets the reject reason.
   *
   * @return string|null
   *   The reject reason or NULL if not set.
   */
  public function getRejectReason(): ?string;

  /**
   * Sets the reject reason.
   *
   * @param string|null $reason
   *   The reject reason or NULL.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setRejectReason(?string $reason): OauthClientRequestInterface;

  /**
   * Gets the OAuth Client Request creation timestamp.
   *
   * @return int
   *   Creation timestamp of the OAuth Client Request.
   */
  public function getCreatedTime(): int;

  /**
   * Sets the OAuth Client Request creation timestamp.
   *
   * @param int $timestamp
   *   The OAuth Client Request creation timestamp.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface
   *   The called OAuth Client Request entity.
   */
  public function setCreatedTime(int $timestamp): OauthClientRequestInterface;

}
