<?php

namespace Drupal\oauth_client\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\NotificationSenderInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Defines hooks for the OAuth Client module.
 */
class OauthClientNotificationHooks {

  public function __construct(
    #[Autowire(service: 'oauth_client.notifications')]
    protected NotificationSenderInterface $notificationSender,
  ) {}

  /**
   * Implements hook_entity_insert().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('oauth_client_request_insert')]
  public function entityInsert(OauthClientRequestInterface $entity): void {
    if ($entity->isStatus(OauthClientRequestStatus::Pending)) {
      $this->notificationSender->handleRequestNotification($entity);
    }
  }

  /**
   * Implements hook_entity_update().
   *
   * @phpstan-ignore-next-line as Attribute class Drupal\Core\Hook\Attribute\Hook does not exist.
   */
  #[Hook('oauth_client_request_update')]
  public function entityUpdate(OauthClientRequestInterface $entity): void {
    $original = $entity->original;
    if (!$original instanceof OauthClientRequestInterface) {
      return;
    }

    if (!$original->isStatus(OauthClientRequestStatus::Pending)) {
      // Only care about transitions from Pending to another state.
      return;
    }

    // Skip if the original didn't have a client but the current one does.
    // This indicates a recursive save from the consumer hook adding the client.
    if (!$original->getClient() && $entity->getClient()) {
      return;
    }

    if ($entity->isStatus(OauthClientRequestStatus::Active) && $original->isStatus(OauthClientRequestStatus::Pending)) {
      $this->notificationSender->handleApprovalNotification($entity);
    }
    elseif ($entity->isStatus(OauthClientRequestStatus::Rejected) && $original->isStatus(OauthClientRequestStatus::Pending)) {
      $this->notificationSender->handleRejectionNotification($entity);
    }
  }

}
