<?php

declare(strict_types=1);

namespace Drupal\Tests\oauth_client\Functional;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Url;
use Drupal\oauth_client\Entity\OauthClientRequest;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\oauth_client\Permission\OauthClientPermissions;
use Drupal\simple_oauth\Entity\Oauth2Scope;
use Drupal\Tests\BrowserTestBase;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\RequestOptions;
use Psr\Http\Message\ResponseInterface;

/**
 * @covers \Drupal\oauth_client\Controller\OauthClientUserController
 * @covers \Drupal\oauth_client\Form\OauthClientEditForm
 * @covers \Drupal\oauth_client\Entity\OauthClientRequest
 * @covers \Drupal\oauth_client\Form\OauthClientRequestForm
 * @covers \Drupal\oauth_client\Form\OauthClientRequestDeleteForm
 * @covers \Drupal\oauth_client\Access\OauthClientRequestAccessControlHandler
 * @group oauth_client
 */
class OauthClientTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['block', 'oauth_client', 'views'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The HTTP client to use for making requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->generateKeys();
    $this->httpClient = \Drupal::service('http_client_factory')
      ->fromOptions(['base_uri' => $this->baseUrl]);

    Oauth2Scope::create(['name' => 'scope1'])->save();
    Oauth2Scope::create(['name' => 'scope2'])->save();

    OauthClientRequestType::create([
      'id' => 'type1',
      'label' => 'Type 1',
      'description' => 'Description of type 1',
      'grant_type' => ['client_credentials' => ['enabled' => TRUE]],
      'scope' => 'scope1',
    ])->save();

    OauthClientRequestType::create([
      'id' => "type2",
      'label' => "Type 2",
      'description' => 'Description of type 2',
      'grant_type' => ['client_credentials' => ['enabled' => TRUE]],
      'scope' => 'scope2',
    ])->save();

    $this->createUser(name: 'regular');
    $this->createUser(
      permissions: [
        OauthClientPermissions::getUserPermission('type1'),
        OauthClientPermissions::getUserPermission('type2'),
      ],
      name: 'qualified',
    );
    $this->createUser(
      permissions: [
        'request type1 oauth2 client',
        'request type2 oauth2 client',
      ],
      name: 'otherQualified');
    $this->createUser(permissions: ['administer oauth clients'], name: 'administrator');
    $this->createUser(permissions: ['manage oauth clients'], name: 'manager');

    $this->drupalPlaceBlock('page_title_block');
    $this->drupalPlaceBlock('local_tasks_block');
    $this->drupalPlaceBlock('local_actions_block');
  }

  /**
   * Tests the OAuth2 client request workflow.
   */
  public function testUserJourney(): void {
    $assert = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Make the bundle entity invalid.
    OauthClientRequestType::load('type1')
      ->set('grant_types', NULL)
      ->set('scope', NULL)
      ->save();

    // User with no permissions cannot create requests.
    $regular = user_load_by_name('regular');
    $this->drupalLogin($regular);
    $this->drupalGet(Url::fromRoute('oauth_client.user', ['user' => $regular->id()]));
    $assert->statusCodeEquals(403);
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.add_page', ['user' => $regular->id()]));
    $assert->statusCodeEquals(403);
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.add_form', [
      'user' => $regular->id(),
      'oauth_client_request_type' => 'type1',
    ]));
    $assert->statusCodeEquals(403);

    // User with permissions cannot create requests for 'type1' because the
    // bundle is malformed. This could happen after a scope entity deletion.
    // @see \Drupal\oauth_client\Entity\OauthClientRequestType::onDependencyRemoval()
    $qualified = user_load_by_name('qualified');
    $uid = $qualified->id();
    $this->drupalLogin($qualified);
    $this->drupalGet(Url::fromRoute('entity.user.edit_form', [
      'user' => $uid,
    ]));
    $page->clickLink('OAuth2 client requests');
    $assert->statusCodeEquals(200);
    $page->clickLink('Request client');

    // Because only 'Type 2' is available, the 'add page' redirects to the form.
    $assert->addressEquals("user/$uid/edit/oauth/add/type2");
    $assert->pageTextContains('Request a new Type 2 client');

    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.add_form', [
      'user' => $uid,
      'oauth_client_request_type' => 'type1',
    ]));
    $assert->statusCodeEquals(403);

    // Fix the bundle entity.
    OauthClientRequestType::load('type1')
      ->set('grant_types', ['client_credentials' => ['status' => TRUE]])
      ->set('scope', 'scope1')
      ->save();

    // User with permissions is now able to make requests.
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.add_page', ['user' => $uid]));
    $assert->pageTextContains('Request a new client');
    $assert->linkExists('Type 1');
    $assert->pageTextContains('Description of type 1');
    $assert->linkExists('Type 2');
    $assert->pageTextContains('Description of type 2');

    $page->clickLink('Type 1');
    $assert->pageTextContains('Request a new Type 1 client');
    // Assert that the request type label and description are displayed.
    $assert->pageTextContains('Type 1');
    $assert->pageTextContains('Description of type 1');
    $page->fillField('Label', 'My mobile app');
    $page->pressButton('Save');

    $assert->statusMessageContains('Request reason field is required.', 'error');
    $page->fillField('Request reason', 'I want to build a mobile app that consumes your APIs');
    $page->fillField('Redirect URI', 'http://example.com');
    $page->pressButton('Save');

    $assert->addressEquals("user/$uid/edit/oauth_requests");
    $assert->statusMessageContains('New OAuth client request My mobile app has been created.', 'status');

    $request = $this->getLatestRequestByAuthor($uid);
    $this->assertSame('My mobile app', $request->label());
    $this->assertSame('I want to build a mobile app that consumes your APIs', $request->getRequestReason());
    $this->assertSame($uid, $request->getOwnerId());
    $this->assertSame(user_load_by_name('qualified')->id(), $request->getUser()->id());
    $this->assertSame([['value' => 'http://example.com']], $request->get('redirect')->getValue());
    // Regular users always create pending new requests.
    $this->assertTrue($request->isStatus(OauthClientRequestStatus::Pending));
    $this->assertNull($request->getClient());

    // User cannot edit their request.
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.canonical', [
      'user' => $uid,
      'oauth_client_request' => $request->id(),
    ]));
    $assert->statusCodeEquals(403);

    // A user tries to access other user's section. It fails.
    $this->drupalLogin(user_load_by_name('otherQualified'));
    $this->drupalGet(Url::fromRoute('oauth_client.user', ['user' => $uid]));
    $assert->statusCodeEquals(403);

    // An administrator tries to access other user's section.
    $this->drupalLogin(user_load_by_name('administrator'));
    $this->drupalGet(Url::fromRoute('oauth_client.user', ['user' => $uid]));
    $assert->statusCodeEquals(200);

    // A manager tries to access other user's section.
    $this->drupalLogin(user_load_by_name('manager'));
    $this->drupalGet(Url::fromRoute('oauth_client.user', ['user' => $uid]));
    $assert->statusCodeEquals(200);

    // Manager is able to edit the request.
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.canonical', [
      'user' => $uid,
      'oauth_client_request' => $request->id(),
    ]));
    $assert->pageTextContains('Edit the client request My mobile app');
    // Assert that the request type label and description are displayed on edit.
    $assert->pageTextContains('Type 1');
    $assert->pageTextContains('Description of type 1');
    $assert->fieldDisabled('Label');
    $assert->fieldDisabled('User');
    $assert->fieldDisabled('Request reason');
    $assert->pageTextContains('Current status: Pending');

    $page->selectFieldOption('status', 'active');
    $page->pressButton('Save');

    $assert->statusMessageContains('The OAuth client request My mobile app has been updated.', 'status');

    $request = $this->container->get('entity_type.manager')
      ->getStorage('oauth_client_request')
      ->loadUnchanged($request->id());

    $consumer = $request->getClient();
    $this->assertNotEmpty($consumer->get('client_id')->getValue());
    $this->assertEquals([['value' => 'client_credentials']], $consumer->get('grant_types')->getValue());
    $this->assertEquals([['scope_id' => 'scope1']], $consumer->get('scopes')->getValue());
    $this->assertNotEmpty($consumer->get('secret')->getValue());

    // Switch back to the user that created the request.
    $this->drupalLogin(user_load_by_name('qualified'));

    // The user cannot delete its own request which was once activated.
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.delete_form', [
      'user' => $uid,
      'oauth_client_request' => $request->id(),
    ]));
    $assert->statusCodeEquals(403);

    // Set the request back to 'Pending'. This cannot happen in real life, but
    // we're forcing here for the sake of the test.
    $request = $this->getLatestRequestByAuthor($uid);
    $request->setStatus(OauthClientRequestStatus::Pending)->save();

    // Users they can delete their own request.
    $this->drupalGet(Url::fromRoute('entity.oauth_client_request.delete_form', [
      'user' => $uid,
      'oauth_client_request' => $request->id(),
    ]));
    $assert->pageTextContains('Are you sure you want to delete the OAuth2 client request My mobile app?');
    $assert->pageTextContains('This action cannot be undone.');
    $page->pressButton('Delete');
    $assert->statusMessageContains('he OAuth2 client request My mobile app has been deleted.', 'status');
    $this->assertNull($this->getLatestRequestByAuthor($uid));
  }

  /**
   * Returns the latest request created by a given user.
   *
   * @param int|string $uid
   *   The user ID.
   *
   * @return \Drupal\oauth_client\Entity\OauthClientRequestInterface|null
   *   The latest request created by a given user or NULL.
   */
  protected function getLatestRequestByAuthor(int|string $uid): ?OauthClientRequestInterface {
    $ids = \Drupal::entityQuery('oauth_client_request')->accessCheck(FALSE)
      ->condition('uid', $uid)
      ->range(0, 1)
      ->sort('id', 'DESC')
      ->execute();
    if (!$ids) {
      return NULL;
    }

    return OauthClientRequest::load(reset($ids));
  }

  /**
   * Asserts that a token from the request is ok.
   *
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client to use.
   * @param string $secret
   *   The client secret to use.
   */
  protected function assertOauth2TokenOk(ConsumerInterface $client, string $secret): void {
    $response = $this->tokenRequest($client, $secret);

    $this->assertSame(200, $response->getStatusCode());
    $this->assertSame('OK', $response->getReasonPhrase());
    $content = (string) $response->getBody();

    $data = json_decode($content, TRUE);
    $this->assertSame('Bearer', $data['token_type']);
    $this->assertSame(300, $data['expires_in']);
    $accessToken = $data['access_token'];

    $response = $this->httpClient->get(Url::fromRoute('oauth2_token.user_debug')
      ->setOption('query', ['_format' => 'json'])
      ->setAbsolute()->toString(), [
        RequestOptions::HEADERS => [
          'Authorization' => 'Bearer ' . $accessToken,
        ],
      ]);

    $this->assertSame(200, $response->getStatusCode());
    $this->assertSame('OK', $response->getReasonPhrase());
    $content = (string) $response->getBody();
    $data = json_decode($content, TRUE);

    $this->assertEquals(user_load_by_name('qualified')->id(), $data['id']);
    $this->assertSame($accessToken, $data['token'][0]);
    $this->assertEquals(user_load_by_name('qualified')->getRoles(), $data['roles']);
  }

  /**
   * Asserts that a token from the request is unauthorized.
   *
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client to use.
   * @param string $secret
   *   The client secret to use.
   */
  protected function assertOauth2TokenUnauthorized(ConsumerInterface $client, string $secret): void {
    $response = $this->tokenRequest($client, $secret);
    $this->assertSame(401, $response->getStatusCode());
    $this->assertSame('Unauthorized', $response->getReasonPhrase());
  }

  /**
   * Makes a token request.
   *
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client to use.
   * @param string $secret
   *   The client secret to use.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response object.
   */
  protected function tokenRequest(ConsumerInterface $client, string $secret): ResponseInterface {
    $url = Url::fromRoute('oauth2_token.token')->setAbsolute()->toString();

    $options = [
      RequestOptions::HEADERS => [
        'Content-Type' => 'application/x-www-form-urlencoded',
        'Accept' => 'application/json',
      ],
      RequestOptions::FORM_PARAMS => [
        'grant_type' => 'client_credentials',
        'client_id' => $client->getClientId(),
        'client_secret' => $secret,
      ],
      RequestOptions::HTTP_ERRORS => FALSE,
    ];

    return $this->httpClient->post($url, $options);
  }

  /**
   * Tests the admin OAuth2 client request view.
   */
  public function testAdminView(): void {
    $assert = $this->assertSession();

    // Create some test requests.
    $qualified = user_load_by_name('qualified');
    $request1 = OauthClientRequest::create([
      'label' => 'Admin View Test 1',
      'type' => 'type1',
      'user' => $qualified->id(),
      'status' => OauthClientRequestStatus::Pending->value,
      'request_reason' => 'Testing admin view',
    ]);
    $request1->save();

    $request2 = OauthClientRequest::create([
      'label' => 'Admin View Test 2',
      'type' => 'type2',
      'user' => $qualified->id(),
      'status' => OauthClientRequestStatus::Active->value,
      'request_reason' => 'Another admin test',
    ]);
    $request2->save();

    // Anonymous users should not have access.
    $this->drupalGet(Url::fromRoute('view.oauth_client_request.admin_page'));
    $assert->statusCodeEquals(403);

    // Regular users without permission should not have access.
    $this->drupalLogin(user_load_by_name('regular'));
    $this->drupalGet(Url::fromRoute('view.oauth_client_request.admin_page'));
    $assert->statusCodeEquals(403);

    // Manager with 'manage oauth clients' permission should have access.
    $this->drupalLogin(user_load_by_name('manager'));
    $this->drupalGet(Url::fromRoute('view.oauth_client_request.admin_page'));
    $assert->statusCodeEquals(200);

    // Verify the page title and content.
    $assert->pageTextContains('OAuth2 client requests');
    $assert->pageTextContains('Admin View Test 1');
    $assert->pageTextContains('Admin View Test 2');

    // Verify status values are displayed.
    $assert->pageTextContains('Pending');
    $assert->pageTextContains('Active');

    // Verify scopes are displayed.
    $assert->pageTextContains('scope1');
    $assert->pageTextContains('scope2');

    // Verify exposed filters are present.
    $assert->fieldExists('Scope');
    $assert->fieldExists('Status');
  }

  /**
   * Generates OAuth2 keys and configures the module to use them.
   */
  protected function generateKeys(): void {
    \Drupal::service('simple_oauth.key.generator')->generateKeys(DRUPAL_ROOT);

    $this->config('simple_oauth.settings')
      ->set('private_key', DRUPAL_ROOT . '/private.key')
      ->set('public_key', DRUPAL_ROOT . '/public.key')
      ->save();
  }

}
