<?php

declare(strict_types=1);

namespace src\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\oauth_client\Entity\OauthClientRequest;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Entity\OauthClientRequestType;
use Drupal\oauth_client\Entity\OauthClientRequestTypeInterface;
use Drupal\oauth_client\Permission\OauthClientPermissions;
use Drupal\oauth_client\Service\OauthClientRequestTypeHelper;
use Drupal\simple_oauth\Entity\Oauth2Scope;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Kernel tests for the OAuth Client module.
 *
 * @group oauth_client
 */
class OauthClientKernelTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected bool $usesSuperUserAccessPolicy = FALSE;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'consumers',
    'file',
    'image',
    'oauth_client',
    'options',
    'serialization',
    'simple_oauth',
    'system',
    'user',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('consumer');
    $this->installEntitySchema('user');
    $this->installEntitySchema('oauth_client_request');
    $this->installConfig(['oauth_client', 'simple_oauth', 'user', 'system']);

    Oauth2Scope::create(['name' => 'scope1'])->save();
    OauthClientRequestType::create([
      'id' => "type1",
      'label' => "Type 1",
      'grant_type' => ['client_credentials' => ['enabled' => TRUE]],
      'scope' => 'scope1',
    ])->save();

    $this->createUser(
      permissions: [OauthClientPermissions::getUserPermission('type1')],
      name: 'qualified',
    );
  }

  /**
   * @covers \Drupal\oauth_client\Entity\OauthClientRequestType::calculateDependencies
   * @covers \Drupal\oauth_client\Entity\OauthClientRequestType::onDependencyRemoval
   * @covers \Drupal\oauth_client\Service\OauthClientRequestTypeHelper::isOauthClientRequestTypeValid
   * @covers \Drupal\oauth_client\Access\OauthClientRequestAccessControlHandler::checkCreateAccess
   */
  public function testScopeDependency(): void {
    $accessHandler = $this->container->get('entity_type.manager')
      ->getAccessControlHandler('oauth_client_request');
    $helper = $this->container->get(OauthClientRequestTypeHelper::class);

    $this->assertSame(Oauth2Scope::load('scope1')->id(), OauthClientRequestType::load('type1')->getScopeId());
    $this->assertTrue($helper->isOauthClientRequestTypeValid(OauthClientRequestType::load('type1')));
    $this->assertTrue($accessHandler->createAccess('type1', user_load_by_name('qualified')));

    // Delete the scope and assert that the type was deleted too.
    Oauth2Scope::load('scope1')->delete();
    $accessHandler->resetCache();
    $this->assertNull(OauthClientRequestType::load('type1'));
    $this->assertFalse($accessHandler->createAccess('type1', user_load_by_name('qualified')));

    // Recreate the scope and type.
    Oauth2Scope::create(['name' => 'scope1'])->save();
    OauthClientRequestType::create([
      'id' => 'type1',
      'label' => 'Type ',
      'scope' => 'scope1',
    ])->save();

    // Permissions are lost because of dependencies. Recreating the user.
    $account = $this->createUser([OauthClientPermissions::getUserPermission('type1')]);

    // Create content for 'type1' type to check that the bundle is preserved.
    $request = OauthClientRequest::create([
      'type' => 'type1',
      'user' => $account,
    ]);
    $request->save();

    // Delete the scope and assert that the type was preserved without scope.
    Oauth2Scope::load('scope1')->delete();
    $type = OauthClientRequestType::load('type1');
    $this->assertInstanceOf(OauthClientRequestTypeInterface::class, $type);
    $this->assertNull($type->getScope());

    // The invalid bundle exists, but it can't be used to create content.
    $this->assertFalse($helper->isOauthClientRequestTypeValid(OauthClientRequestType::load('type1')));
    $this->assertFalse($accessHandler->createAccess('type1', $account));
  }

  /**
   * @covers \Drupal\oauth_client\Entity\OauthClientRequestType
   */
  public function testOauthClientRequestCrud(): void {
    $request = OauthClientRequest::create([
      'type' => 'type1',
      'label' => 'Test Request',
      'uid' => user_load_by_name('qualified'),
      'user' => user_load_by_name('qualified'),
      'request_reason' => 'Testing purpose',
    ]);
    $request->save();

    $this->assertSame('Test Request', $request->label());
    $this->assertSame(OauthClientRequestStatus::Pending, $request->getStatus());
    $this->assertTrue($request->isStatus(OauthClientRequestStatus::Pending));

    // Test rejection.
    $request->setStatus(OauthClientRequestStatus::Rejected);
    $request->setRejectReason('Insufficient justification');
    $request->save();
    $this->assertSame(OauthClientRequestStatus::Rejected, $request->getStatus());
    $this->assertSame('Insufficient justification', $request->getRejectReason());

    // Test approval.
    $request->setStatus(OauthClientRequestStatus::Active);
    $request->save();
    $this->assertSame(OauthClientRequestStatus::Active, $request->getStatus());
  }

}
