<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Access reusable code.
 */
trait AccessTrait {

  /**
   * Returns managers access.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   Whether the account has access as manager.
   */
  protected function getManagerAccess(AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIfHasPermissions($account, [
      'administer oauth clients',
      'manage oauth clients',
    ], 'OR');
  }

  /**
   * Checks whether the passed user account is a manager.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to be checked.
   *
   * @return bool
   *   The check result.
   */
  protected function isManager(AccountInterface $account): bool {
    return $this->getManagerAccess($account)->isAllowed();
  }

  /**
   * Returns oauth client access result object based on account and route match.
   *
   * Access is allowed if current account is an OAuth client manager or if the
   * account is the same as which can be found in the route slugs at "user".
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check access of.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The (current) route match to check the access with.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function managerOrSelfAccessResult(AccountInterface $account, RouteMatchInterface $routeMatch): AccessResultInterface {
    // Account is a manager.
    return $this->getManagerAccess($account)
      // Account is the same user as in the route.
      ->orIf(AccessResult::allowedIf(
        $routeMatch->getRawParameters()->has('user') &&
        $routeMatch->getRawParameter('user') == $account->id()
      ));
  }

}
