<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\oauth_client\Entity\OauthClientRequestInterface;
use Drupal\oauth_client\Entity\OauthClientRequestStatus;
use Drupal\oauth_client\Permission\OauthClientPermissions;
use Drupal\oauth_client\Service\OauthClientRequestTypeHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the access control handler for the OAuth Client Request entity type.
 */
class OauthClientRequestAccessControlHandler extends EntityAccessControlHandler implements EntityHandlerInterface {

  use AccessTrait;

  public function __construct(
    EntityTypeInterface $entityType,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly OauthClientRequestTypeHelper $typeHelper,
  ) {
    parent::__construct($entityType);
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): static {
    return new static(
      $entity_type,
      $container->get('entity_type.manager'),
      $container->get(OauthClientRequestTypeHelper::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultInterface {
    assert($entity instanceof OauthClientRequestInterface);

    return match ($operation) {
      'update' => $this->checkUpdateAccess($entity, $account),
      'delete' => $this->checkDeleteAccess($entity, $account),
      default => AccessResult::neutral(),
    };
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL): AccessResultInterface {
    $bundleIsValid = $this->oauth2ClientRequestTypeIsValid($entity_bundle);

    $access = $this->getManagerAccess($account);

    if ($access->andIf($bundleIsValid)->isAllowed()) {
      return $access;
    }

    $permission = OauthClientPermissions::getUserPermission($entity_bundle);
    $access = $access->orIf(AccessResult::allowedIfHasPermission($account, $permission));

    return $access->andIf($bundleIsValid);
  }

  /**
   * Checks update access for an OAuth Client Request entity.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestInterface $entity
   *   The OAuth Client Request entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function checkUpdateAccess(OauthClientRequestInterface $entity, AccountInterface $account): AccessResultInterface {
    // Users with manage permission can edit requests.
    $access = $this->getManagerAccess($account);

    // Can only edit if status is Pending or Rejected.
    $statusAccess = AccessResult::allowedIf(
      $entity->isStatus(OauthClientRequestStatus::Pending) ||
      $entity->isStatus(OauthClientRequestStatus::Rejected)
    );

    // Chain access checks: manager AND editable status.
    return $access->andIf($statusAccess)->addCacheableDependency($entity);
  }

  /**
   * Checks delete access for an OAuth Client Request entity.
   *
   * @param \Drupal\oauth_client\Entity\OauthClientRequestInterface $entity
   *   The OAuth Client Request entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function checkDeleteAccess(OauthClientRequestInterface $entity, AccountInterface $account): AccessResultInterface {
    // Managers can delete any request.
    $managerAccess = $this->getManagerAccess($account);

    // Regular users can only delete their own pending requests.
    $ownerAccess = AccessResult::allowedIf($entity->getOwnerId() == $account->id())
      ->cachePerUser();

    $pendingAccess = AccessResult::allowedIf($entity->isStatus(OauthClientRequestStatus::Pending));

    $permission = OauthClientPermissions::getUserPermission($entity->get('type')->entity);
    $typeAccess = AccessResult::allowedIfHasPermission($account, $permission);

    // Chain access checks: manager OR (owner AND pending AND type permission).
    return $managerAccess->orIf(
      $ownerAccess->andIf($pendingAccess)->andIf($typeAccess)
    )->addCacheableDependency($entity);
  }

  /**
   * Checks if the OAuth2 Client Request type entity is valid.
   *
   * @param string|null $bundle
   *   The OAuth2 Client Request type ID.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The result of the check, containing cache metadata.
   */
  protected function oauth2ClientRequestTypeIsValid(?string $bundle): AccessResultInterface {
    if (!$bundle) {
      return AccessResult::neutral();
    }
    $storage = $this->entityTypeManager->getStorage('oauth_client_request_type');
    if (!$entity = $storage->load($bundle)) {
      return AccessResult::neutral();
    }

    // Check bundle integrity.
    return AccessResult::allowedIf($this->typeHelper->isOauthClientRequestTypeValid($entity))
      ->addCacheableDependency($entity);
  }

}
