<?php

declare(strict_types=1);

namespace Drupal\oauth_client\EventSubscriber;

use Drupal\oauth_client\Entity\NotificationType;
use Drupal\oauth_client\Event\NotificationEvent;
use Drupal\oauth_client\NotificationSenderInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for OAuth client notifications.
 */
final class OauthClientNotificationSubscriber implements EventSubscriberInterface {

  /**
   * Constructs an OauthClientNotificationSubscriber object.
   */
  public function __construct(
    private readonly NotificationSenderInterface $oauthClientNotificationSender,
  ) {}

  /**
   * Reacts to an entity CRUD event.
   *
   * @param \Drupal\oauth_client\Event\NotificationEvent $event
   *   The event object.
   */
  public function onEntityCrud(NotificationEvent $event): void {
    $request = $event->getOauthClientRequest();
    if ($event->getAction() === NotificationType::Request) {
      $this->oauthClientNotificationSender->handleRequestNotification($request);
    }
    elseif ($event->getAction() === NotificationType::Approve) {
      $this->oauthClientNotificationSender->handleApprovalNotification($request);
    }
    elseif ($event->getAction() === NotificationType::Reject) {
      $this->oauthClientNotificationSender->handleRejectionNotification($request);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      NotificationEvent::class => 'onEntityCrud',
    ];
  }

}
