<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\consumers\Entity\ConsumerInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\oauth_client\Access\AccessTrait;
use Drupal\user\UserInterface;

/**
 * Trait to check access for OAuth client forms.
 */
trait FormAccessTrait {

  use AccessTrait;

  /**
   * Checks access for a given operation on a client by a user.
   *
   * @param string $operation
   *   The operation to check.
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   * @param \Drupal\consumers\Entity\ConsumerInterface $client
   *   The OAuth2 client.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result object.
   */
  protected function checkAccess(string $operation, UserInterface $user, ConsumerInterface $client): AccessResultInterface {
    $access = $client->access(operation: $operation, return_as_object: TRUE);
    if ($access->isAllowed()) {
      return $access;
    }

    $access = $access->orIf($this->getManagerAccess($this->currentUser()));
    if ($access->isAllowed()) {
      return $access;
    }

    $isConfidential = !$client->get('confidential')->isEmpty() && $client->get('confidential')->first()->getValue()['value'];
    $isCurrentUser = $user->id() == $this->currentUser()->id();
    $clientBelongsToUser = $client->get('user_id')->target_id == $user->id();

    $clientAccess = AccessResult::allowedIf($isConfidential && $isCurrentUser && $clientBelongsToUser)
      ->addCacheableDependency($client)
      ->addCacheableDependency($user);

    return $access->orIf($clientAccess);
  }

}
