<?php

declare(strict_types=1);

namespace Drupal\oauth_client\Form;

use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;

/**
 * Provides a form for deleting an OAuth Client Request entity.
 */
class OauthClientRequestDeleteForm extends ContentEntityDeleteForm {

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete the OAuth client request %label?', [
      '%label' => $this->getEntity()->getLabel(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    $entity = $this->getEntity();
    return new Url('oauth_client.user', [
      'user' => $entity->getUser()->id(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Delete request');
  }

  /**
   * {@inheritdoc}
   */
  protected function getDeletionMessage(): TranslatableMarkup {
    $entity = $this->getEntity();
    return $this->t('The OAuth client request %label has been deleted.', [
      '%label' => $entity->getLabel(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $entity = $this->getEntity();
    parent::submitForm($form, $form_state);

    $form_state->setRedirect('oauth_client.user', [
      'user' => $entity->getUser()
        ->id(),
    ]);
  }

  /**
   * Gets the title for the delete form.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   *
   * @return string
   *   The page title.
   */
  public static function title(RouteMatchInterface $route_match): string {
    $entity = $route_match->getParameter('oauth_client_request');
    return (string) t('Delete OAuth client request: @label', [
      '@label' => $entity ? $entity->getLabel() : '',
    ]);
  }

}
